# IBrandBiz – Outline & Sections (Core Build)

Here’s the expanded spec + scaffold with more structured templates and section support.

---

## Default Plan Outline (Must-Have Sections)
1. **Executive Summary**
2. **Company Overview**
   - Mission & Vision
   - Legal Structure
   - Founding Team
3. **Products & Services**
   - Product/Service Description
   - Value Proposition
   - Roadmap
4. **Market Analysis**
   - Industry Overview
   - Target Market (TAM/SAM/SOM)
   - Competitive Landscape
   - SWOT Analysis (Template)
5. **Marketing & Sales**
   - Customer Personas (Template)
   - Go-to-Market Strategy
   - Pricing Model (Template)
   - Sales Channels
6. **Operations Plan**
   - Key Activities
   - Resources & Suppliers
   - Technology Stack
7. **Organization & Management**
   - Org Chart
   - Roles & Responsibilities
   - Advisors
8. **Financial Plan**
   - Revenue Model
   - Cost Structure
   - 3-Year Projections (tables)
   - Break-even Analysis
   - Funding Requirements

---

## Templates (Fill-in-the-Blank)

### SWOT Analysis
- **Strengths:** [List internal advantages]
- **Weaknesses:** [List internal challenges]
- **Opportunities:** [List external trends/opportunities]
- **Threats:** [List external risks]

### Customer Persona
- **Name:** [Persona label]
- **Demographics:** Age, gender, location
- **Goals:** [Primary objectives]
- **Frustrations:** [Pain points]
- **Buying Triggers:** [Motivators]

### TAM / SAM / SOM
- **TAM (Total Addressable Market):** [Global demand]
- **SAM (Serviceable Available Market):** [Reachable market]
- **SOM (Serviceable Obtainable Market):** [Market share goal]

### Pricing Table
| Tier | Features | Price |
|------|----------|-------|
| Free | Basic access | $0 |
| Pro | Premium features | $49/mo |
| Enterprise | Custom features & support | Contact us |

### Porter’s Five Forces
1. Threat of New Entrants
2. Bargaining Power of Suppliers
3. Bargaining Power of Buyers
4. Threat of Substitutes
5. Industry Rivalry

---

## Drag-and-Drop Editor Features
- Reorder sections and sub-sections (tree-based)
- Add new section or template block
- Duplicate/delete sections
- Inline rename

---

## AI Prompt Hooks
- **Outline Generator:**
  > “Generate a structured business plan outline for [industry], with 6–10 sections and optional subsections.”

- **Section Drafting:**
  > “Write a [tone: Professional/Friendly/Bold/Minimal] draft for the section [name], using inputs: [user notes]. Return markdown paragraphs only.”

---

## Next Dev Steps
- Persist outline (localStorage → API)
- UI rendering with shadcn (Cards, Tree View)
- Live preview of templates → markdown → rendered prose
- Wire export (PDF/DOCX/Google Docs)

---

/* =========================
   V2 ADDITIONS – nested subsections, persistence, more templates
   ========================= */

/* =========================
   File: src/state/usePlanStore.v2.ts
   ========================= */
import { create as createZ } from "zustand";
import { nanoid as nid } from "nanoid";
import type { PlanState, PlanSection, SectionKind, SectionId } from "../types/plan";

const V2_STORAGE_KEY = "ibrandbiz.plan.v2";

const V2_DEFAULT_SECTIONS: PlanSection[] = [
  { id: nid(), kind: "executive-summary", title: "Executive Summary", content: "" },
  { id: nid(), kind: "company-overview", title: "Company Overview", content: "" },
  { id: nid(), kind: "products-services", title: "Products & Services", content: "" },
  { id: nid(), kind: "market-analysis", title: "Market Analysis", content: "" },
  { id: nid(), kind: "marketing-sales", title: "Marketing & Sales", content: "" },
  { id: nid(), kind: "operations", title: "Operations Plan", content: "" },
  { id: nid(), kind: "org-management", title: "Organization & Management", content: "" },
  { id: nid(), kind: "financials", title: "Financial Plan", content: "" },
];

function v2Load(): PlanState {
  try {
    const raw = localStorage.getItem(V2_STORAGE_KEY);
    if (raw) return JSON.parse(raw) as PlanState;
  } catch {}
  return { planId: nid(), title: "Untitled Business Plan", sections: V2_DEFAULT_SECTIONS };
}

type V2Store = PlanState & {
  addSection(kind?: SectionKind, title?: string): SectionId;
  addChildSection(parentId: SectionId, title?: string): SectionId | null;
  removeSection(id: SectionId): void;
  removeChildSection(parentId: SectionId, childId: SectionId): void;
  reorderSections(idsInOrder: SectionId[]): void;
  reorderChildSections(parentId: SectionId, idsInOrder: SectionId[]): void;
  updateSection(id: SectionId, patch: Partial<PlanSection>): void;
  updateChildSection(parentId: SectionId, childId: SectionId, patch: Partial<PlanSection>): void;
  duplicateSection(id: SectionId): SectionId | null;
  resetToDefault(): void;
};

export const usePlanStoreV2 = createZ<V2Store>((set, get) => ({
  ...v2Load(),
  addSection: (kind = "custom", title = "New Section") => {
    const sec: PlanSection = { id: nid(), kind, title, content: "", children: [] };
    set((s) => ({ sections: [...s.sections, sec] }));
    v2Persist();
    return sec.id;
  },
  addChildSection: (parentId, title = "New Subsection") => {
    set((s) => ({
      sections: s.sections.map((sec) =>
        sec.id === parentId
          ? { ...sec, children: [...(sec.children || []), { id: nid(), kind: "custom", title, content: "" }] }
          : sec
      ),
    }));
    v2Persist();
    const p = get().sections.find((x) => x.id === parentId);
    const last = p?.children?.[p.children.length - 1];
    return last?.id || null;
  },
  removeSection: (id) => { set((s) => ({ sections: s.sections.filter((x) => x.id !== id) })); v2Persist(); },
  removeChildSection: (parentId, childId) => {
    set((s) => ({
      sections: s.sections.map((sec) => sec.id === parentId ? { ...sec, children: (sec.children||[]).filter((c)=>c.id!==childId)}: sec)
    }));
    v2Persist();
  },
  reorderSections: (ids) => { set((s) => ({ sections: ids.map((id) => s.sections.find((x)=>x.id===id)!).filter(Boolean) })); v2Persist(); },
  reorderChildSections: (parentId, ids) => {
    set((s) => ({
      sections: s.sections.map((sec) => {
        if (sec.id !== parentId) return sec;
        const current = sec.children || [];
        const out = ids.map((id) => current.find((c)=>c.id===id)!).filter(Boolean);
        return { ...sec, children: out };
      })
    }));
    v2Persist();
  },
  updateSection: (id, patch) => { set((s)=>({ sections: s.sections.map((sec)=>sec.id===id?{...sec, ...patch}:sec) })); v2Persist(); },
  updateChildSection: (parentId, childId, patch) => {
    set((s)=>({ sections: s.sections.map((sec)=> sec.id!==parentId? sec : { ...sec, children: (sec.children||[]).map((c)=> c.id===childId?{...c, ...patch}:c) }) }));
    v2Persist();
  },
  duplicateSection: (id) => {
    const src = get().sections.find((x)=>x.id===id); if (!src) return null;
    const dupe: PlanSection = { ...src, id: nid(), title: `${src.title} (Copy)` };
    set((s)=>({ sections: [...s.sections, dupe] })); v2Persist(); return dupe.id;
  },
  resetToDefault: () => {
    const state = { planId: nid(), title: "Untitled Business Plan", sections: V2_DEFAULT_SECTIONS };
    set(state); try { localStorage.setItem(V2_STORAGE_KEY, JSON.stringify(state)); } catch {}
  },
}));

function v2Persist(){
  try {
    const { planId, title, sections } = usePlanStoreV2.getState();
    localStorage.setItem(V2_STORAGE_KEY, JSON.stringify({ planId, title, sections }));
  } catch {}
}

/* =========================
   File: src/templates/structured.v2.ts
   ========================= */
import type { StructuredTemplate } from "../types/plan";

export const V2_TEMPLATES: StructuredTemplate[] = [
  {
    key: "porter5",
    name: "Porter’s Five Forces",
    description: "Analyze industry competitiveness.",
    fields: [
      { id: "threat_new", label: "Threat of New Entrants", type: "textarea" },
      { id: "bargain_sup", label: "Bargaining Power of Suppliers", type: "textarea" },
      { id: "bargain_buy", label: "Bargaining Power of Buyers", type: "textarea" },
      { id: "threat_sub", label: "Threat of Substitutes", type: "textarea" },
      { id: "rivalry", label: "Industry Rivalry", type: "textarea" },
    ],
  },
  {
    key: "tam_sam_som",
    name: "TAM / SAM / SOM",
    description: "Market sizing overview.",
    fields: [
      { id: "tam", label: "TAM (Total Addressable Market)", type: "textarea", placeholder: "Size, assumptions, sources" },
      { id: "sam", label: "SAM (Serviceable Available Market)", type: "textarea" },
      { id: "som", label: "SOM (Serviceable Obtainable Market)", type: "textarea" },
      { id: "sources", label: "Sources / Citations", type: "textarea" },
    ],
  },
  {
    key: "pricing_table",
    name: "Pricing Table",
    description: "List tiers or SKUs with price & notes.",
    fields: [
      { id: "rows", label: "Rows (Name – Price – Notes)", type: "textarea", placeholder: "Basic – $19/mo – For starters
Pro – $49/mo – Most popular
Enterprise – $199/mo – Custom SLAs" },
    ],
  },
];

export function renderV2TemplateMarkdown(key: string, data: Record<string,string>): string {
  if (key === "porter5") return `### Porter’s Five Forces

**Threat of New Entrants**
${data.threat_new||"-"}

**Bargaining Power of Suppliers**
${data.bargain_sup||"-"}

**Bargaining Power of Buyers**
${data.bargain_buy||"-"}

**Threat of Substitutes**
${data.threat_sub||"-"}

**Industry Rivalry**
${data.rivalry||"-"}`;
  if (key === "tam_sam_som") return `### Market Size: TAM / SAM / SOM

**TAM**
${data.tam||"-"}

**SAM**
${data.sam||"-"}

**SOM**
${data.som||"-"}

**Sources**
${data.sources||"-"}`;
  if (key === "pricing_table") {
    const rows = (data.rows||"").split(/
+/).filter(Boolean);
    const table = ["| Name | Price | Notes |", "|---|---:|---|"]
      .concat(rows.map((r)=>{ const [name="", price="", notes=""] = r.split(" – "); return `| ${name.trim()} | ${price.trim()} | ${notes.trim()} |`; }))
      .join("
");
    return `### Pricing

${table}`;
  }
  return "";
}

/* =========================
   File: src/components/OutlineEditor.v2.tsx
   ========================= */
import React from "react";
import { DndContext, closestCenter, PointerSensor, useSensor, useSensors } from "@dnd-kit/core";
import { arrayMove, SortableContext, verticalListSortingStrategy } from "@dnd-kit/sortable";
import { CSS } from "@dnd-kit/utilities";
import { useSortable } from "@dnd-kit/sortable";
import { usePlanStoreV2 } from "../state/usePlanStore.v2";

function Row({ id, title, parentId }: { id: string; title: string; parentId?: string }) {
  const { attributes, listeners, setNodeRef, transform, transition, isDragging } = useSortable({ id });
  const update = usePlanStoreV2((s)=> s.updateSection);
  const updateChild = usePlanStoreV2((s)=> s.updateChildSection);
  const remove = usePlanStoreV2((s)=> s.removeSection);
  const removeChild = usePlanStoreV2((s)=> s.removeChildSection);
  const addChild = usePlanStoreV2((s)=> s.addChildSection);

  const style: React.CSSProperties = { transform: CSS.Transform.toString(transform), transition, opacity: isDragging?0.6:1 };
  const onRename = (val:string)=> parentId ? updateChild(parentId, id, { title: val }) : update(id, { title: val });
  const onDelete = ()=> parentId ? removeChild(parentId, id) : remove(id);

  return (
    <div ref={setNodeRef} style={style} className="flex items-center justify-between rounded-xl border p-2">
      <div className="flex items-center gap-3">
        <button {...attributes} {...listeners} className="cursor-grab">≡</button>
        <input className="border-none outline-none bg-transparent text-base" defaultValue={title} onBlur={(e)=> onRename(e.currentTarget.value)} />
      </div>
      <div className="flex gap-2">
        {!parentId && (<button onClick={()=> addChild(id)} className="px-2 py-1 rounded-lg border">Add Subsection</button>)}
        <button onClick={onDelete} className="px-2 py-1 rounded-lg border text-red-600">Delete</button>
      </div>
    </div>
  );
}

export function OutlineEditorV2(){
  const sections = usePlanStoreV2((s)=> s.sections);
  const reorder = usePlanStoreV2((s)=> s.reorderSections);
  const reorderChild = usePlanStoreV2((s)=> s.reorderChildSections);
  const sensors = useSensors(useSensor(PointerSensor));

  function onDragEndTop(event:any){
    const { active, over } = event; if (!over || active.id===over.id) return;
    const ids = sections.map((s)=> s.id); const oldIndex = ids.indexOf(active.id); const newIndex = ids.indexOf(over.id);
    reorder(arrayMove(ids, oldIndex, newIndex));
  }
  return (
    <div className="space-y-3">
      <DndContext sensors={sensors} collisionDetection={closestCenter} onDragEnd={onDragEndTop}>
        <SortableContext items={sections.map((s)=> s.id)} strategy={verticalListSortingStrategy}>
          {sections.map((s)=> (
            <div key={s.id} className="rounded-2xl border p-3">
              <Row id={s.id} title={s.title} />

              {(s.children||[]).length>0 && (
                <ChildList parentId={s.id} />
              )}
            </div>
          ))}
        </SortableContext>
      </DndContext>
    </div>
  );

  function ChildList({ parentId }:{ parentId: string }){
    const parent = sections.find((x)=> x.id===parentId);
    const children = parent?.children || [];

    function onDragEndChild(event:any){
      const { active, over } = event; if (!over || active.id===over.id) return;
      const ids = children.map((c)=> c.id); const oldIndex = ids.indexOf(active.id); const newIndex = ids.indexOf(over.id);
      reorderChild(parentId, arrayMove(ids, oldIndex, newIndex));
    }
    return (
      <div className="mt-3 ml-7">
        <DndContext sensors={sensors} collisionDetection={closestCenter} onDragEnd={onDragEndChild}>
          <SortableContext items={children.map((c)=> c.id)} strategy={verticalListSortingStrategy}>
            <div className="space-y-2">
              {children.map((c)=> (<Row key={c.id} id={c.id} title={c.title} parentId={parentId} />))}
            </div>
          </SortableContext>
        </DndContext>
      </div>
    );
  }
}

/* =========================
   File: src/components/StructuredForm.v2.tsx
   ========================= */
import React, { useState } from "react";
import { V2_TEMPLATES, renderV2TemplateMarkdown } from "../templates/structured.v2";
import { usePlanStoreV2 } from "../state/usePlanStore.v2";

export function StructuredFormV2({ targetSectionId }:{ targetSectionId: string }){
  const [selectedKey, setSelectedKey] = useState<string>(V2_TEMPLATES[0].key);
  const [data, setData] = useState<Record<string,string>>({});
  const update = usePlanStoreV2((s)=> s.updateSection);

  const tpl = V2_TEMPLATES.find((t)=> t.key===selectedKey)!;
  function insert(){
    const md = renderV2TemplateMarkdown(tpl.key, data);
    update(targetSectionId, (prev=> ({ content: `${(prev as any)?.content? (prev as any).content + "

" : ""}${md}` })) as any);
  }

  return (
    <div className="space-y-3">
      <select value={selectedKey} onChange={(e)=> setSelectedKey(e.target.value)} className="border rounded-xl p-2">
        {V2_TEMPLATES.map((t)=> (<option key={t.key} value={t.key}>{t.name}</option>))}
      </select>
      <div className="grid gap-2">
        {tpl.fields.map((f)=> (
          <label key={f.id} className="grid gap-1">
            <span className="text-sm text-gray-600">{f.label}</span>
            {f.type === "textarea" ? (
              <textarea rows={3} className="rounded-xl border p-2" placeholder={f.placeholder} onChange={(e)=> setData((d)=> ({...d, [f.id]: e.target.value}))} />
            ) : (
              <input className="rounded-xl border p-2" placeholder={f.placeholder} onChange={(e)=> setData((d)=> ({...d, [f.id]: e.target.value}))} />
            )}
          </label>
        ))}
      </div>
      <button onClick={insert} className="rounded-2xl border px-3 py-2">Insert into Section</button>
    </div>
  );
}

/* =========================
   File: src/App.v2.tsx (integration sketch)
   ========================= */
import React from "react";
import { OutlineEditorV2 } from "./components/OutlineEditor.v2";
import { usePlanStoreV2 } from "./state/usePlanStore.v2";
import { StructuredFormV2 } from "./components/StructuredForm.v2";

export default function AppV2(){
  const sections = usePlanStoreV2((s)=> s.sections);
  const update = usePlanStoreV2((s)=> s.updateSection);
  const first = sections[0];

  return (
    <div className="grid md:grid-cols-2 gap-6 p-6">
      <div>
        <h2 className="text-xl font-semibold mb-3">Outline (V2)</h2>
        <OutlineEditorV2 />
      </div>
      <div>
        <h2 className="text-xl font-semibold mb-3">Structured Templates (V2)</h2>
        {first && <StructuredFormV2 targetSectionId={first.id} />}
        {first && (
          <div className="mt-6">
            <h3 className="font-medium mb-2">Live Preview (first section content)</h3>
            <textarea className="w-full h-48 rounded-xl border p-3" value={first.content} onChange={(e)=> update(first.id, { content: e.target.value })} />
          </div>
        )}
      </div>
    </div>
  );
}


/* =========================
   FINANCIALS – Deterministic Inputs + 12‑Month P&L
   ========================= */

/* =========================
   File: src/types/finance.ts
   ========================= */
export type Currency = "USD" | "EUR" | "GBP" | "CAD" | "AUD";

export interface FinanceInputs {
  currency: Currency;
  startMonth: string; // YYYY-MM (e.g., "2025-10")
  pricePerUnit: number; // average selling price
  unitsPerMonth: number[]; // length 12, units sold per month
  cogsPercent: number; // 0..100
  fixedMonthlyExpenses: number; // rent, tools, etc.
  variableExpensePerUnit?: number; // optional extra variable cost
  startingCash?: number; // for runway calc
}

export interface MonthlyRow {
  month: string; // e.g., "Oct 2025"
  revenue: number;
  cogs: number;
  grossProfit: number;
  opex: number; // fixedMonthlyExpenses
  profit: number; // grossProfit - opex
  cumulativeProfit: number;
}

export interface FinanceOutputs {
  rows: MonthlyRow[]; // 12 months
  totalRevenue: number;
  totalProfit: number;
  breakevenMonthIndex: number | null; // 0..11 or null
  runwayMonths: number | null; // if startingCash provided
}

/* =========================
   File: src/lib/financeCalc.ts
   ========================= */
import type { FinanceInputs, FinanceOutputs, MonthlyRow } from "../types/finance";

const MONTH_NAMES = ["Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"];

function formatMonthLabel(ym: string, i: number) {
  // ym = YYYY-MM
  const [y0, m0] = ym.split("-").map(Number);
  const base = new Date(y0, (m0 - 1) + i, 1);
  return `${MONTH_NAMES[base.getMonth()]} ${base.getFullYear()}`;
}

export function computePAndL(input: FinanceInputs): FinanceOutputs {
  const rows: MonthlyRow[] = [];
  let cumulative = 0;
  const price = Number(input.pricePerUnit) || 0;
  const cogsPct = Math.min(Math.max(input.cogsPercent, 0), 100) / 100;
  const varPerUnit = Number(input.variableExpensePerUnit || 0);
  const opex = Number(input.fixedMonthlyExpenses) || 0;

  for (let i = 0; i < 12; i++) {
    const units = Number(input.unitsPerMonth[i] || 0);
    const revenue = units * price;
    const cogs = revenue * cogsPct + units * varPerUnit;
    const grossProfit = revenue - cogs;
    const profit = grossProfit - opex;
    cumulative += profit;

    rows.push({
      month: formatMonthLabel(input.startMonth, i),
      revenue, cogs, grossProfit, opex, profit, cumulativeProfit: cumulative,
    });
  }

  const totalRevenue = rows.reduce((s, r) => s + r.revenue, 0);
  const totalProfit = rows.reduce((s, r) => s + r.profit, 0);

  // breakeven = first month where cumulativeProfit >= 0
  let breakevenMonthIndex: number | null = null;
  for (let i = 0; i < rows.length; i++) {
    if (rows[i].cumulativeProfit >= 0) { breakevenMonthIndex = i; break; }
  }

  let runwayMonths: number | null = null;
  if (input.startingCash != null) {
    let cash = input.startingCash;
    let m = 0;
    while (m < 60) { // cap at 5 years
      const idx = Math.min(m, rows.length - 1);
      cash += rows[idx].profit;
      if (cash <= 0) { runwayMonths = m + 1; break; }
      m++;
    }
    if (runwayMonths == null) runwayMonths = 60; // survival beyond horizon
  }

  return { rows, totalRevenue, totalProfit, breakevenMonthIndex, runwayMonths };
}

/* =========================
   File: src/state/useFinanceStore.ts
   ========================= */
import { create } from "zustand";
import type { FinanceInputs, FinanceOutputs } from "../types/finance";
import { computePAndL } from "../lib/financeCalc";

const KEY = "ibrandbiz.finance.v1";

export type FinanceStore = {
  inputs: FinanceInputs;
  outputs: FinanceOutputs | null;
  setInputs: (patch: Partial<FinanceInputs>) => void;
  recompute: () => void;
  reset: () => void;
};

const DEFAULT_INPUTS: FinanceInputs = {
  currency: "USD",
  startMonth: "2025-10",
  pricePerUnit: 49,
  unitsPerMonth: [10,12,15,18,22,26,30,35,40,46,52,60],
  cogsPercent: 30,
  fixedMonthlyExpenses: 3000,
  variableExpensePerUnit: 0,
  startingCash: 5000,
};

function load(): FinanceInputs {
  try { const raw = localStorage.getItem(KEY); if (raw) return JSON.parse(raw); } catch {}
  return DEFAULT_INPUTS;
}

export const useFinanceStore = create<FinanceStore>((set, get) => ({
  inputs: load(),
  outputs: null,
  setInputs: (patch) => {
    set((s) => ({ inputs: { ...s.inputs, ...patch } }));
    persist(get().inputs);
  },
  recompute: () => {
    const out = computePAndL(get().inputs);
    set({ outputs: out });
  },
  reset: () => {
    set({ inputs: DEFAULT_INPUTS, outputs: null });
    persist(DEFAULT_INPUTS);
  },
}));

function persist(i: FinanceInputs) { try { localStorage.setItem(KEY, JSON.stringify(i)); } catch {} }

/* =========================
   File: src/components/FinanceInputs.tsx
   ========================= */
import React, { useEffect } from "react";
import { useFinanceStore } from "../state/useFinanceStore";

export function FinanceInputsPanel(){
  const { inputs, setInputs, recompute, outputs } = useFinanceStore();
  useEffect(()=>{ recompute(); }, []);

  const updateUnits = (idx:number, val:number) => {
    const arr = [...inputs.unitsPerMonth];
    arr[idx] = val;
    setInputs({ unitsPerMonth: arr });
    recompute();
  };

  return (
    <div className="space-y-4">
      <div className="grid grid-cols-2 gap-3">
        <label className="grid gap-1">
          <span className="text-sm text-gray-600">Start Month (YYYY-MM)</span>
          <input className="border rounded-xl p-2" value={inputs.startMonth} onChange={(e)=>{setInputs({ startMonth: e.target.value }); recompute();}} />
        </label>
        <label className="grid gap-1">
          <span className="text-sm text-gray-600">Currency</span>
          <select className="border rounded-xl p-2" value={inputs.currency} onChange={(e)=>{setInputs({ currency: e.target.value as any }); recompute();}}>
            {(["USD","EUR","GBP","CAD","AUD"] as const).map(c=> <option key={c} value={c}>{c}</option>)}
          </select>
        </label>
        <label className="grid gap-1">
          <span className="text-sm text-gray-600">Price per Unit</span>
          <input type="number" className="border rounded-xl p-2" value={inputs.pricePerUnit} onChange={(e)=>{setInputs({ pricePerUnit: Number(e.target.value) }); recompute();}} />
        </label>
        <label className="grid gap-1">
          <span className="text-sm text-gray-600">COGS %</span>
          <input type="number" className="border rounded-xl p-2" value={inputs.cogsPercent} onChange={(e)=>{setInputs({ cogsPercent: Number(e.target.value) }); recompute();}} />
        </label>
        <label className="grid gap-1">
          <span className="text-sm text-gray-600">Fixed Monthly Expenses</span>
          <input type="number" className="border rounded-xl p-2" value={inputs.fixedMonthlyExpenses} onChange={(e)=>{setInputs({ fixedMonthlyExpenses: Number(e.target.value) }); recompute();}} />
        </label>
        <label className="grid gap-1">
          <span className="text-sm text-gray-600">Variable Expense per Unit (optional)</span>
          <input type="number" className="border rounded-xl p-2" value={inputs.variableExpensePerUnit||0} onChange={(e)=>{setInputs({ variableExpensePerUnit: Number(e.target.value) }); recompute();}} />
        </label>
        <label className="grid gap-1">
          <span className="text-sm text-gray-600">Starting Cash (optional)</span>
          <input type="number" className="border rounded-xl p-2" value={inputs.startingCash||0} onChange={(e)=>{setInputs({ startingCash: Number(e.target.value) }); recompute();}} />
        </label>
      </div>

      <div className="mt-2">
        <span className="text-sm text-gray-600">Units per Month (12)
        </span>
        <div className="grid grid-cols-6 gap-2 mt-2">
          {Array.from({length:12}).map((_,i)=> (
            <input key={i} type="number" className="border rounded-lg p-2" value={inputs.unitsPerMonth[i]||0} onChange={(e)=> updateUnits(i, Number(e.target.value))} />
          ))}
        </div>
      </div>

      {outputs && (
        <div className="mt-4 grid gap-2">
          <div className="rounded-xl border p-3 text-sm">
            <div>Total Revenue: <strong>{inputs.currency} {outputs.totalRevenue.toLocaleString()}</strong></div>
            <div>Total Profit: <strong>{inputs.currency} {outputs.totalProfit.toLocaleString()}</strong></div>
            <div>Breakeven: <strong>{outputs.breakevenMonthIndex!=null ? outputs.rows[outputs.breakevenMonthIndex].month : "Not in 12 months"}</strong></div>
            {inputs.startingCash!=null && <div>Runway (est.): <strong>{outputs.runwayMonths} mo</strong></div>}
          </div>

          <div className="overflow-x-auto">
            <table className="w-full border-collapse border border-border">
              <thead className="bg-muted">
                <tr>
                  <th className="border p-2 text-left">Month</th>
                  <th className="border p-2 text-left">Revenue</th>
                  <th className="border p-2 text-left">COGS</th>
                  <th className="border p-2 text-left">Gross Profit</th>
                  <th className="border p-2 text-left">Opex</th>
                  <th className="border p-2 text-left">Profit</th>
                  <th className="border p-2 text-left">Cum. Profit</th>
                </tr>
              </thead>
              <tbody>
                {outputs.rows.map((r,idx)=> (
                  <tr key={idx}>
                    <td className="border p-2">{r.month}</td>
                    <td className="border p-2">{inputs.currency} {r.revenue.toLocaleString()}</td>
                    <td className="border p-2">{inputs.currency} {r.cogs.toLocaleString()}</td>
                    <td className="border p-2">{inputs.currency} {r.grossProfit.toLocaleString()}</td>
                    <td className="border p-2">{inputs.currency} {r.opex.toLocaleString()}</td>
                    <td className="border p-2">{inputs.currency} {r.profit.toLocaleString()}</td>
                    <td className="border p-2">{inputs.currency} {r.cumulativeProfit.toLocaleString()}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>
      )}
    </div>
  );
}

/* =========================
   File: src/components/FinanceSidebar.tsx
   ========================= */
import React from "react";
import { FinanceInputsPanel } from "./FinanceInputs";

export function FinanceSidebar(){
  return (
    <aside className="space-y-4">
      <h3 className="text-lg font-semibold">Financials</h3>
      <p className="text-sm text-muted-foreground">Enter a few assumptions to auto-generate your 12‑month P&L.</p>
      <FinanceInputsPanel />
    </aside>
  );
}

/* =========================
   AI WRITING ASSISTANT – Inline actions per section
   ========================= */

/* =========================
   File: src/types/ai.ts
   ========================= */
export type ToneOption = "Formal" | "Friendly" | "Persuasive" | "Inspirational" | "Technical";
export type AiAction = "generate" | "rephrase" | "expand" | "summarize";

export interface AiJob {
  action: AiAction;
  tone?: ToneOption;
  sectionTitle: string;
  companyName?: string;
  context?: string; // existing text or plan brief
}

/* =========================
   File: src/services/ai/aiClient.ts
   ========================= */
import type { AiJob } from "../../types/ai";

// Replace with your actual API route that calls GPT‑5 / GPT‑5‑mini
export async function callAi(job: AiJob): Promise<string> {
  const res = await fetch("/api/ai/section", {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify(job),
  });
  if (!res.ok) throw new Error("AI request failed");
  const data = await res.json();
  return data.text as string;
}

/* =========================
   File: src/components/AIButton.tsx
   ========================= */
import React, { useState } from "react";
import type { AiAction, ToneOption } from "../types/ai";
import { callAi } from "../services/ai/aiClient";

const TONES: ToneOption[] = ["Formal","Friendly","Persuasive","Inspirational","Technical"];

export function AIButton({
  sectionTitle,
  companyName,
  getContext,
  onApply,
}: {
  sectionTitle: string;
  companyName?: string;
  getContext: () => string; // current text
  onApply: (text: string) => void;
}){
  const [open, setOpen] = useState(false);
  const [loading, setLoading] = useState(false);
  const [tone, setTone] = useState<ToneOption>("Formal");
  const [action, setAction] = useState<AiAction>("generate");
  const [error, setError] = useState<string | null>(null);

  async function run(){
    setError(null); setLoading(true);
    try {
      const text = await callAi({ action, tone, sectionTitle, companyName, context: getContext() });
      onApply(text);
      setOpen(false);
    } catch(e:any){ setError(e?.message || "AI failed"); }
    finally { setLoading(false); }
  }

  return (
    <div className="relative inline-block">
      <button className="rounded-lg border px-2 py-1 text-sm" onClick={()=> setOpen((v)=> !v)}>Write with AI</button>
      {open && (
        <div className="absolute z-20 mt-2 w-64 rounded-xl border bg-white p-3 shadow-lg">
          <div className="grid gap-2">
            <label className="grid gap-1">
              <span className="text-xs text-gray-600">Action</span>
              <select className="border rounded-md p-1" value={action} onChange={(e)=> setAction(e.target.value as any)}>
                <option value="generate">Generate</option>
                <option value="rephrase">Rephrase</option>
                <option value="expand">Expand</option>
                <option value="summarize">Summarize</option>
              </select>
            </label>
            <label className="grid gap-1">
              <span className="text-xs text-gray-600">Tone</span>
              <select className="border rounded-md p-1" value={tone} onChange={(e)=> setTone(e.target.value as any)}>
                {TONES.map(t=> <option key={t} value={t}>{t}</option>)}
              </select>
            </label>
            {error && <div className="text-xs text-red-600">{error}</div>}
            <button className="rounded-md border px-2 py-1 text-sm" onClick={run} disabled={loading}>{loading? "Thinking…" : "Apply"}</button>
          </div>
        </div>
      )}
    </div>
  );
}

/* =========================
   File: src/components/SectionEditor.tsx
   ========================= */
import React from "react";
import { usePlanStoreV2 } from "../state/usePlanStore.v2";
import { AIButton } from "./AIButton";

export function SectionEditor({ sectionId }:{ sectionId: string }){
  const section = usePlanStoreV2((s)=> s.sections.find(x=> x.id===sectionId));
  const update = usePlanStoreV2((s)=> s.updateSection);
  if (!section) return null;

  return (
    <div className="space-y-2">
      <div className="flex items-center justify-between">
        <h4 className="font-semibold">{section.title}</h4>
        <AIButton
          sectionTitle={section.title}
          companyName={"IBrandBiz"}
          getContext={() => section.content}
          onApply={(text) => update(sectionId, { content: text })}
        />
      </div>
      <textarea
        className="w-full h-48 rounded-xl border p-3"
        value={section.content}
        onChange={(e)=> update(sectionId, { content: e.target.value })}
        placeholder={`Write your ${section.title} here…`}
      />
    </div>
  );
}

/* =========================
   File: src/pages/Builder.tsx (integration sketch)
   ========================= */
import React from "react";
import { OutlineEditorV2 } from "../components/OutlineEditor.v2";
import { usePlanStoreV2 } from "../state/usePlanStore.v2";
import { SectionEditor } from "../components/SectionEditor";
import { FinanceSidebar } from "../components/FinanceSidebar";

export default function Builder(){
  const sections = usePlanStoreV2((s)=> s.sections);
  const first = sections[0];

  return (
    <div className="grid lg:grid-cols-[1.1fr_0.9fr] gap-8 p-6">
      <div className="space-y-6">
        <h2 className="text-xl font-semibold">Outline</h2>
        <OutlineEditorV2 />

        {first && (
          <div className="mt-6">
            <SectionEditor sectionId={first.id} />
          </div>
        )}
      </div>

      <div className="space-y-6">
        <FinanceSidebar />
      </div>
    </div>
  );
}

/* =========================
   File: src/pages/api/ai/section.ts (Next.js example)
   ========================= */
// import type { NextApiRequest, NextApiResponse } from "next";
// import type { AiJob } from "../../../types/ai";
// import OpenAI from "openai";
// const client = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });
//
// export default async function handler(req: NextApiRequest, res: NextApiResponse){
//   if (req.method !== "POST") return res.status(405).end();
//   const job = req.body as AiJob;
//   const sys = "You are an expert business plan writer.";
//   const user = buildPrompt(job);
//   const r = await client.responses.create({ model: "gpt-5-mini", input: [{ role: "system", content: sys }, { role: "user", content: user }] });
//   // @ts-ignore
//   const text = r.output_text || r.output[0]?.content[0]?.text || "";
//   res.json({ text });
// }
//
// function buildPrompt(job: AiJob){
//   const base = `Section: ${job.sectionTitle}
Company: ${job.companyName||""}
Tone: ${job.tone||"Formal"}`;
//   if (job.action === "generate") return base + `
Write a concise, skimmable section (180–250 words).`;
//   if (job.action === "rephrase") return base + `
Rephrase the following to be clearer and more impactful, preserving meaning:

${job.context||""}`;
//   if (job.action === "expand") return base + `
Expand the following with 2–3 crisp paragraphs and a short bullet list:

${job.context||""}`;
//   if (job.action === "summarize") return base + `
Summarize the following into 3–5 bullets and a 2‑sentence close:

${job.context||""}`;
//   return base;
// }


/* =========================
   FINANCIAL TOOLS – Charts, Editable Tables, CSV Import, AI Insights
   ========================= */

/* =========================
   File: src/types/finance.v2.ts
   ========================= */
export interface ExpenseRow { id: string; name: string; amount: number; }
export interface FinanceInputsV2 extends FinanceInputs {
  expenses: ExpenseRow[]; // optional granular opex rows (summed into fixedMonthlyExpenses)
}

/* =========================
   File: src/state/useFinanceStore.v2.ts
   ========================= */
import { create } from "zustand";
import { nanoid } from "nanoid";
import type { FinanceOutputs } from "../types/finance";
import type { FinanceInputsV2, ExpenseRow } from "../types/finance.v2";
import { computePAndL } from "../lib/financeCalc";

const KEY_V2 = "ibrandbiz.finance.v2";

const DEFAULT_EXPENSES: ExpenseRow[] = [
  { id: nanoid(), name: "Rent & Utilities", amount: 1200 },
  { id: nanoid(), name: "Software & Tools", amount: 400 },
  { id: nanoid(), name: "Marketing", amount: 800 },
  { id: nanoid(), name: "Misc", amount: 200 },
];

const DEFAULT_V2: FinanceInputsV2 = {
  currency: "USD",
  startMonth: "2025-10",
  pricePerUnit: 49,
  unitsPerMonth: [10,12,15,18,22,26,30,35,40,46,52,60],
  cogsPercent: 30,
  fixedMonthlyExpenses: DEFAULT_EXPENSES.reduce((s,x)=> s + x.amount, 0),
  variableExpensePerUnit: 0,
  startingCash: 5000,
  expenses: DEFAULT_EXPENSES,
};

function loadV2(): FinanceInputsV2 { try { const raw = localStorage.getItem(KEY_V2); if (raw) return JSON.parse(raw); } catch {} return DEFAULT_V2; }

export type FinanceStoreV2 = {
  inputs: FinanceInputsV2;
  outputs: FinanceOutputs | null;
  setInputs: (patch: Partial<FinanceInputsV2>) => void;
  setExpenses: (rows: ExpenseRow[]) => void;
  recompute: () => void;
  reset: () => void;
};

export const useFinanceStoreV2 = create<FinanceStoreV2>((set, get) => ({
  inputs: loadV2(),
  outputs: null,
  setInputs: (patch) => { set((s)=> ({ inputs: { ...s.inputs, ...patch } })); persist(get().inputs); },
  setExpenses: (rows) => {
    set((s)=> ({ inputs: { ...s.inputs, expenses: rows, fixedMonthlyExpenses: rows.reduce((sum,r)=> sum + (Number(r.amount)||0), 0) } }));
    persist(get().inputs);
  },
  recompute: () => { const out = computePAndL(get().inputs); set({ outputs: out }); },
  reset: () => { set({ inputs: DEFAULT_V2, outputs: null }); persist(DEFAULT_V2); },
}));

function persist(i: FinanceInputsV2){ try { localStorage.setItem(KEY_V2, JSON.stringify(i)); } catch {} }

/* =========================
   File: src/components/ExpensesTable.tsx
   ========================= */
import React from "react";
import { useFinanceStoreV2 } from "../state/useFinanceStore.v2";
import { nanoid } from "nanoid";

export function ExpensesTable(){
  const { inputs, setExpenses, recompute } = useFinanceStoreV2();
  const rows = inputs.expenses || [];

  const update = (id:string, field:"name"|"amount", value:string) => {
    const next = rows.map(r=> r.id===id ? { ...r, [field]: field==="amount"? Number(value): value } : r);
    setExpenses(next); recompute();
  };
  const addRow = () => { setExpenses([...rows, { id: nanoid(), name: "New Expense", amount: 0 }]); };
  const delRow = (id:string) => { setExpenses(rows.filter(r=> r.id!==id)); recompute(); };

  return (
    <div className="space-y-2">
      <div className="flex items-center justify-between">
        <h4 className="font-medium">Operating Expenses</h4>
        <button className="rounded-md border px-2 py-1 text-sm" onClick={addRow}>Add Row</button>
      </div>
      <div className="overflow-x-auto">
        <table className="w-full border-collapse border border-border text-sm">
          <thead className="bg-muted">
            <tr>
              <th className="border p-2 text-left">Name</th>
              <th className="border p-2 text-right">Amount / mo</th>
              <th className="border p-2"></th>
            </tr>
          </thead>
          <tbody>
            {rows.map(r=> (
              <tr key={r.id}>
                <td className="border p-2"><input className="w-full outline-none" value={r.name} onChange={(e)=> update(r.id, "name", e.target.value)} /></td>
                <td className="border p-2 text-right"><input type="number" className="w-full text-right outline-none" value={r.amount} onChange={(e)=> update(r.id, "amount", e.target.value)} /></td>
                <td className="border p-2 text-right"><button className="rounded-md border px-2 py-1" onClick={()=> delRow(r.id)}>Delete</button></td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
      <div className="text-right text-sm">Fixed Monthly Total: <strong>{inputs.currency} {rows.reduce((s,r)=> s + (Number(r.amount)||0),0).toLocaleString()}</strong></div>
    </div>
  );
}

/* =========================
   File: src/components/FinanceCharts.tsx
   ========================= */
import React from "react";
import { useFinanceStoreV2 } from "../state/useFinanceStore.v2";
import { AreaChart, Area, XAxis, YAxis, Tooltip, ResponsiveContainer, LineChart, Line, Legend, BarChart, Bar } from "recharts";

export function FinanceCharts(){
  const { outputs, inputs } = useFinanceStoreV2();
  if (!outputs) return null;
  const data = outputs.rows.map(r=> ({ month: r.month, revenue: r.revenue, profit: r.profit, cash: (inputs.startingCash||0) + r.cumulativeProfit }));

  return (
    <div className="grid gap-6">
      <div className="rounded-2xl border p-3">
        <h4 className="font-medium mb-2">Revenue vs Profit</h4>
        <div className="h-56">
          <ResponsiveContainer width="100%" height="100%">
            <LineChart data={data}>
              <XAxis dataKey="month" hide/>
              <YAxis/>
              <Tooltip formatter={(v:number)=> `${inputs.currency} ${v.toLocaleString()}`}/>
              <Legend />
              <Line type="monotone" dataKey="revenue" stroke="#8884d8" dot={false} name="Revenue" />
              <Line type="monotone" dataKey="profit" stroke="#82ca9d" dot={false} name="Profit" />
            </LineChart>
          </ResponsiveContainer>
        </div>
      </div>

      <div className="rounded-2xl border p-3">
        <h4 className="font-medium mb-2">Cash Flow (Ending Cash)</h4>
        <div className="h-56">
          <ResponsiveContainer width="100%" height="100%">
            <AreaChart data={data}>
              <XAxis dataKey="month" hide/>
              <YAxis/>
              <Tooltip formatter={(v:number)=> `${inputs.currency} ${v.toLocaleString()}`}/>
              <Area type="monotone" dataKey="cash" stroke="#8884d8" fill="#8884d8" fillOpacity={0.2} name="Ending Cash" />
            </AreaChart>
          </ResponsiveContainer>
        </div>
      </div>

      <div className="rounded-2xl border p-3">
        <h4 className="font-medium mb-2">Break‑Even (Profit by Month)</h4>
        <div className="h-56">
          <ResponsiveContainer width="100%" height="100%">
            <BarChart data={outputs.rows}>
              <XAxis dataKey="month" hide/>
              <YAxis/>
              <Tooltip formatter={(v:number)=> `${inputs.currency} ${v.toLocaleString()}`}/>
              <Bar dataKey="profit" name="Profit" fill="#82ca9d" />
            </BarChart>
          </ResponsiveContainer>
        </div>
        {outputs.breakevenMonthIndex!=null && (
          <p className="text-sm mt-2">Breakeven in <strong>{outputs.rows[outputs.breakevenMonthIndex].month}</strong>.</p>
        )}
      </div>
    </div>
  );
}

/* =========================
   File: src/components/CsvImport.tsx
   ========================= */
import React from "react";
import Papa from "papaparse";
import { useFinanceStoreV2 } from "../state/useFinanceStore.v2";

/**
 * CSV formats supported:
 * 1) units per month (12 numbers, header optional): month,units
 * 2) expenses table: name,amount
 */
export function CsvImport(){
  const { setInputs, setExpenses, recompute } = useFinanceStoreV2();

  function onFile(e: React.ChangeEvent<HTMLInputElement>, kind:"units"|"expenses"){
    const file = e.target.files?.[0]; if (!file) return;
    Papa.parse(file, {
      header: true,
      skipEmptyLines: true,
      complete: (res:any) => {
        if (kind === "units"){
          const vals: number[] = [];
          (res.data as any[]).forEach(row=> { const v = Number(row.units ?? row.Units ?? row.value ?? row.Value); if (!isNaN(v)) vals.push(v); });
          if (vals.length >= 12){ setInputs({ unitsPerMonth: vals.slice(0,12) }); recompute(); }
        } else {
          const rows = (res.data as any[]).map(r=> ({ name: String(r.name||r.Name||"Expense"), amount: Number(r.amount||r.Amount||0), id: crypto.randomUUID?.() || String(Math.random()) }));
          setExpenses(rows); recompute();
        }
      }
    });
  }

  return (
    <div className="grid gap-2 text-sm">
      <label className="grid gap-1">
        <span className="text-gray-600">Import Units (CSV)</span>
        <input type="file" accept=".csv" onChange={(e)=> onFile(e, "units")} />
      </label>
      <label className="grid gap-1">
        <span className="text-gray-600">Import Expenses (CSV)</span>
        <input type="file" accept=".csv" onChange={(e)=> onFile(e, "expenses")} />
      </label>
    </div>
  );
}

/* =========================
   File: src/components/FinanceInsightsAI.tsx
   ========================= */
import React, { useState } from "react";
import { useFinanceStoreV2 } from "../state/useFinanceStore.v2";
import { callAi } from "../services/ai/aiClient";

export function FinanceInsightsAI(){
  const { inputs, outputs } = useFinanceStoreV2();
  const [text, setText] = useState<string>("");
  const [loading, setLoading] = useState(false);
  const [err, setErr] = useState<string|null>(null);

  async function run(){
    if (!outputs) return;
    setLoading(true); setErr(null);
    const brief = {
      currency: inputs.currency,
      startMonth: inputs.startMonth,
      price: inputs.pricePerUnit,
      units: inputs.unitsPerMonth,
      cogsPercent: inputs.cogsPercent,
      fixedOpex: inputs.fixedMonthlyExpenses,
      startingCash: inputs.startingCash,
      totals: { revenue: outputs.totalRevenue, profit: outputs.totalProfit },
      breakeven: outputs.breakevenMonthIndex!=null ? outputs.rows[outputs.breakevenMonthIndex].month : null,
    };
    const prompt = `You are a sharp financial analyst. Given this 12‑month P&L JSON, answer succinctly: 1) When do we break even? 2) Biggest driver of profit/loss? 3) One actionable lever to improve margin. Output 3 bullets.

${JSON.stringify(brief, null, 2)}`;
    try {
      const out = await callAi({ action: "generate", tone: "Technical", sectionTitle: "Financial Insights", context: prompt });
      setText(out);
    } catch(e:any){ setErr(e?.message||"AI failed"); }
    finally { setLoading(false); }
  }

  return (
    <div className="rounded-2xl border p-3">
      <div className="flex items-center justify-between mb-2">
        <h4 className="font-medium">AI Financial Insights</h4>
        <button className="rounded-md border px-2 py-1 text-sm" onClick={run} disabled={loading || !outputs}>{loading? "Thinking…" : "Analyze"}</button>
      </div>
      {err && <div className="text-sm text-red-600">{err}</div>}
      <pre className="whitespace-pre-wrap text-sm">{text}</pre>
    </div>
  );
}

/* =========================
   File: src/components/FinancialsPanel.tsx
   ========================= */
import React, { useEffect } from "react";
import { useFinanceStoreV2 } from "../state/useFinanceStore.v2";
import { FinanceInputsPanel } from "./FinanceInputs";
import { ExpensesTable } from "./ExpensesTable";
import { CsvImport } from "./CsvImport";
import { FinanceCharts } from "./FinanceCharts";
import { FinanceInsightsAI } from "./FinanceInsightsAI";

export function FinancialsPanel(){
  const { recompute } = useFinanceStoreV2();
  useEffect(()=>{ recompute(); }, []);

  return (
    <div className="grid gap-6">
      <FinanceInputsPanel />
      <ExpensesTable />
      <CsvImport />
      <FinanceCharts />
      <FinanceInsightsAI />
    </div>
  );
}

/* =========================
   File: src/pages/Builder.v2.tsx (updated integration)
   ========================= */
import React from "react";
import { OutlineEditorV2 } from "../components/OutlineEditor.v2";
import { usePlanStoreV2 } from "../state/usePlanStore.v2";
import { SectionEditor } from "../components/SectionEditor";
import { FinancialsPanel } from "../components/FinancialsPanel";

export default function BuilderV2(){
  const sections = usePlanStoreV2((s)=> s.sections);
  const first = sections[0];
  return (
    <div className="grid lg:grid-cols-[1.1fr_0.9fr] gap-8 p-6">
      <div className="space-y-6">
        <h2 className="text-xl font-semibold">Outline</h2>
        <OutlineEditorV2 />
        {first && (
          <div className="mt-6">
            <SectionEditor sectionId={first.id} />
          </div>
        )}
      </div>
      <div className="space-y-6">
        <h2 className="text-xl font-semibold">Financial Tools</h2>
        <FinancialsPanel />
      </div>
    </div>
  );
}


/* =========================
   VISUALS & TEMPLATES – SWOT, Persona, TAM/SAM/SOM, Pricing
   ========================= */

/* =========================
   File: src/types/visuals.ts
   ========================= */
export interface SwotCard { id: string; text: string }
export interface SwotData { strengths: SwotCard[]; weaknesses: SwotCard[]; opportunities: SwotCard[]; threats: SwotCard[] }

export interface PersonaData {
  name: string;
  demographics: string;
  goals: string;
  pains: string;
  needs: string;
}

export interface MarketSizeData { tam: string; sam: string; som: string; sources?: string }

export interface PricingRow { name: string; price: string; notes?: string }

/* =========================
   File: src/lib/visualsRender.ts
   ========================= */
import type { SwotData, PersonaData, MarketSizeData, PricingRow } from "../types/visuals";

export function renderSwotMarkdown(d: SwotData){
  const list = (arr: {text:string}[]) => arr.length? arr.map(x=> `- ${x.text}`).join("
") : "-";
  return `### SWOT Analysis

**Strengths**
${list(d.strengths)}

**Weaknesses**
${list(d.weaknesses)}

**Opportunities**
${list(d.opportunities)}

**Threats**
${list(d.threats)}`;
}

export function renderPersonaMarkdown(p: PersonaData){
  return `### Customer Persona: ${p.name||"Unnamed"}

**Demographics**
${p.demographics||"-"}

**Goals**
${p.goals||"-"}

**Pain Points**
${p.pains||"-"}

**Needs**
${p.needs||"-"}`;
}

export function renderMarketSizeMarkdown(m: MarketSizeData){
  return `### Market Size (TAM / SAM / SOM)

**TAM**
${m.tam||"-"}

**SAM**
${m.sam||"-"}

**SOM**
${m.som||"-"}

**Sources**
${m.sources||"-"}`;
}

export function renderPricingMarkdown(rows: PricingRow[]){
  const header = `| Plan | Price | Notes |
|---|---:|---|`;
  const body = rows.map(r=> `| ${r.name} | ${r.price} | ${r.notes||""} |`).join("
");
  return `### Pricing

${header}
${body}`;
}

/* =========================
   File: src/components/SwotBoard.tsx
   ========================= */
import React, { useState } from "react";
import { nanoid } from "nanoid";
import type { SwotData } from "../types/visuals";
import { renderSwotMarkdown } from "../lib/visualsRender";
import { usePlanStoreV2 } from "../state/usePlanStore.v2";

const empty: SwotData = { strengths:[], weaknesses:[], opportunities:[], threats:[] };

function Column({ title, items, onAdd, onEdit, onDel }:{ title:string; items:{id:string;text:string}[]; onAdd:()=>void; onEdit:(id:string,val:string)=>void; onDel:(id:string)=>void }){
  return (
    <div className="rounded-2xl border p-3 grid gap-2">
      <div className="flex items-center justify-between">
        <h5 className="font-medium">{title}</h5>
        <button className="rounded-md border px-2 py-1 text-xs" onClick={onAdd}>Add</button>
      </div>
      {items.length===0 && <div className="text-xs text-muted-foreground">No items yet</div>}
      {items.map(it=> (
        <div key={it.id} className="flex items-center gap-2">
          <input className="flex-1 border rounded-md p-2 text-sm" value={it.text} onChange={(e)=> onEdit(it.id, e.target.value)} />
          <button className="rounded-md border px-2 py-1 text-xs" onClick={()=> onDel(it.id)}>Del</button>
        </div>
      ))}
    </div>
  );
}

export function SwotBoard({ targetSectionId }:{ targetSectionId: string }){
  const [data, setData] = useState<SwotData>(empty);
  const update = usePlanStoreV2((s)=> s.updateSection);

  const add = (k: keyof SwotData) => setData(d=> ({ ...d, [k]: [...d[k], { id: nanoid(), text: "" }] }));
  const edit = (k: keyof SwotData, id: string, val: string) => setData(d=> ({ ...d, [k]: d[k].map(c=> c.id===id? { ...c, text: val } : c) }));
  const del = (k: keyof SwotData, id: string) => setData(d=> ({ ...d, [k]: d[k].filter(c=> c.id!==id) }));

  function insert(){
    const md = renderSwotMarkdown(data);
    update(targetSectionId, (prev=> ({ content: `${(prev as any)?.content? (prev as any).content + "

" : ""}${md}` })) as any);
  }

  return (
    <div className="grid md:grid-cols-2 gap-3">
      <Column title="Strengths" items={data.strengths} onAdd={()=> add("strengths")} onEdit={(id,v)=> edit("strengths", id, v)} onDel={(id)=> del("strengths", id)} />
      <Column title="Weaknesses" items={data.weaknesses} onAdd={()=> add("weaknesses")} onEdit={(id,v)=> edit("weaknesses", id, v)} onDel={(id)=> del("weaknesses", id)} />
      <Column title="Opportunities" items={data.opportunities} onAdd={()=> add("opportunities")} onEdit={(id,v)=> edit("opportunities", id, v)} onDel={(id)=> del("opportunities", id)} />
      <Column title="Threats" items={data.threats} onAdd={()=> add("threats")} onEdit={(id,v)=> edit("threats", id, v)} onDel={(id)=> del("threats", id)} />

      <div className="md:col-span-2 flex justify-end">
        <button className="rounded-2xl border px-3 py-2" onClick={insert}>Insert into Section</button>
      </div>
    </div>
  );
}

/* =========================
   File: src/components/PersonaForm.tsx
   ========================= */
import React, { useState } from "react";
import type { PersonaData } from "../types/visuals";
import { renderPersonaMarkdown } from "../lib/visualsRender";
import { usePlanStoreV2 } from "../state/usePlanStore.v2";

export function PersonaForm({ targetSectionId }:{ targetSectionId: string }){
  const [data, set] = useState<PersonaData>({ name:"", demographics:"", goals:"", pains:"", needs:"" });
  const update = usePlanStoreV2((s)=> s.updateSection);

  const bind = (k: keyof PersonaData) => ({ value: data[k] as string, onChange: (e: any)=> set({ ...data, [k]: e.target.value }) });

  function insert(){
    const md = renderPersonaMarkdown(data);
    update(targetSectionId, (prev=> ({ content: `${(prev as any)?.content? (prev as any).content + "

" : ""}${md}` })) as any);
  }

  return (
    <div className="grid gap-3">
      <label className="grid gap-1">
        <span className="text-sm text-gray-600">Persona Name</span>
        <input className="rounded-xl border p-2" {...bind("name")} placeholder="e.g., Side‑Hustle Sarah" />
      </label>
      <label className="grid gap-1">
        <span className="text-sm text-gray-600">Demographics</span>
        <textarea className="rounded-xl border p-2" rows={2} {...bind("demographics")} />
      </label>
      <label className="grid gap-1">
        <span className="text-sm text-gray-600">Goals</span>
        <textarea className="rounded-xl border p-2" rows={2} {...bind("goals")} />
      </label>
      <label className="grid gap-1">
        <span className="text-sm text-gray-600">Pain Points</span>
        <textarea className="rounded-xl border p-2" rows={2} {...bind("pains")} />
      </label>
      <label className="grid gap-1">
        <span className="text-sm text-gray-600">Needs</span>
        <textarea className="rounded-xl border p-2" rows={2} {...bind("needs")} />
      </label>
      <div className="flex justify-end">
        <button className="rounded-2xl border px-3 py-2" onClick={insert}>Insert into Section</button>
      </div>
    </div>
  );
}

/* =========================
   File: src/components/TemplatesPanel.tsx
   ========================= */
import React, { useState } from "react";
import { SwotBoard } from "./SwotBoard";
import { PersonaForm } from "./PersonaForm";
import { MarketSizeForm } from "./visuals/MarketSizeForm";
import { PricingTableForm } from "./visuals/PricingTableForm";
import { CustomChartBuilder } from "./visuals/CustomChartBuilder";
import { usePlanStoreV2 } from "../state/usePlanStore.v2";

export function TemplatesPanel(){
  const sections = usePlanStoreV2((s)=> s.sections);
  const [targetId, setTarget] = useState<string>(sections[0]?.id || "");
  const [tab, setTab] = useState<"swot"|"persona"|"tam"|"pricing"|"chart">("swot");

  return (
    <div className="grid gap-4">
      <div className="grid gap-1">
        <label className="text-sm text-gray-600">Insert into section</label>
        <select className="border rounded-xl p-2" value={targetId} onChange={(e)=> setTarget(e.target.value)}>
          {sections.map(s=> <option key={s.id} value={s.id}>{s.title}</option>)}
        </select>
      </div>

      <div className="flex flex-wrap gap-2">
        <button className={`rounded-md border px-3 py-1 text-sm ${tab==='swot'?'bg-muted':''}`} onClick={()=> setTab('swot')}>SWOT</button>
        <button className={`rounded-md border px-3 py-1 text-sm ${tab==='persona'?'bg-muted':''}`} onClick={()=> setTab('persona')}>Persona</button>
        <button className={`rounded-md border px-3 py-1 text-sm ${tab==='tam'?'bg-muted':''}`} onClick={()=> setTab('tam')}>TAM/SAM/SOM</button>
        <button className={`rounded-md border px-3 py-1 text-sm ${tab==='pricing'?'bg-muted':''}`} onClick={()=> setTab('pricing')}>Pricing Table</button>
        <button className={`rounded-md border px-3 py-1 text-sm ${tab==='chart'?'bg-muted':''}`} onClick={()=> setTab('chart')}>Custom Chart</button>
      </div>

      {tab==='swot' && <SwotBoard targetSectionId={targetId} />}
      {tab==='persona' && <PersonaForm targetSectionId={targetId} />}
      {tab==='tam' && <MarketSizeForm targetSectionId={targetId} />}
      {tab==='pricing' && <PricingTableForm targetSectionId={targetId} />}
      {tab==='chart' && <CustomChartBuilder targetSectionId={targetId} />}
    </div>
  );
}

/* =========================
   File: src/pages/Builder.v3.tsx (integration with Templates Panel)
   ========================= */
import React from "react";
import { OutlineEditorV2 } from "../components/OutlineEditor.v2";
import { usePlanStoreV2 } from "../state/usePlanStore.v2";
import { SectionEditor } from "../components/SectionEditor";
import { FinancialsPanel } from "../components/FinancialsPanel";
import { TemplatesPanel } from "../components/TemplatesPanel";

export default function BuilderV3(){
  const sections = usePlanStoreV2((s)=> s.sections);
  const first = sections[0];
  return (
    <div className="grid xl:grid-cols-[1.1fr_0.9fr] gap-8 p-6">
      <div className="space-y-6">
        <h2 className="text-xl font-semibold">Outline</h2>
        <OutlineEditorV2 />
        {first && (
          <div className="mt-6">
            <SectionEditor sectionId={first.id} />
          </div>
        )}
      </div>
      <div className="space-y-8">
        <div>
          <h2 className="text-xl font-semibold mb-2">Visual Templates</h2>
          <TemplatesPanel />
        </div>
        <div>
          <h2 className="text-xl font-semibold mb-2">Financial Tools</h2>
          <FinancialsPanel />
        </div>
      </div>
    </div>
  );
}

/* =========================
   File: src/components/visuals/MarketSizeForm.tsx
   ========================= */
import React, { useState } from "react";
import type { MarketSizeData } from "../../types/visuals";
import { renderMarketSizeMarkdown } from "../../lib/visualsRender";
import { usePlanStoreV2 } from "../../state/usePlanStore.v2";
import { PieChart, Pie, Cell, ResponsiveContainer, Tooltip, Legend } from "recharts";

export function MarketSizeForm({ targetSectionId }:{ targetSectionId: string }){
  const [data, set] = useState<MarketSizeData>({ tam:"", sam:"", som:"", sources:"" });
  const update = usePlanStoreV2((s)=> s.updateSection);

  const nums = {
    TAM: Number(String(data.tam).replace(/[^0-9.]/g, "")) || 0,
    SAM: Number(String(data.sam).replace(/[^0-9.]/g, "")) || 0,
    SOM: Number(String(data.som).replace(/[^0-9.]/g, "")) || 0,
  };
  const chartData = [
    { name: "TAM", value: nums.TAM },
    { name: "SAM", value: nums.SAM },
    { name: "SOM", value: nums.SOM },
  ];
  const COLORS = ["#8884d8", "#82ca9d", "#ffc658"];

  function insert(){
    const md = renderMarketSizeMarkdown(data);
    const prevAppender = function(p:any){
      var base = (p && (p as any).content) ? (p as any).content + "

" : "";
      return { content: base + md } as any;
    };
    update(targetSectionId, prevAppender as any);
  }

  return (
    <div className="grid md:grid-cols-2 gap-4">
      <div className="grid gap-3">
        <label className="grid gap-1"><span className="text-sm text-gray-600">TAM</span><input className="rounded-xl border p-2" placeholder="$ market size" value={data.tam} onChange={(e)=> set({ ...data, tam: e.target.value })} /></label>
        <label className="grid gap-1"><span className="text-sm text-gray-600">SAM</span><input className="rounded-xl border p-2" placeholder="$ market size" value={data.sam} onChange={(e)=> set({ ...data, sam: e.target.value })} /></label>
        <label className="grid gap-1"><span className="text-sm text-gray-600">SOM</span><input className="rounded-xl border p-2" placeholder="$ market size" value={data.som} onChange={(e)=> set({ ...data, som: e.target.value })} /></label>
        <label className="grid gap-1"><span className="text-sm text-gray-600">Sources</span><textarea className="rounded-xl border p-2" rows={2} value={data.sources} onChange={(e)=> set({ ...data, sources: e.target.value })} /></label>
        <div className="flex justify-end"><button className="rounded-2xl border px-3 py-2" onClick={insert}>Insert into Section</button></div>
      </div>
      <div className="rounded-2xl border p-3">
        <h5 className="font-medium mb-2">Market Size Visual</h5>
        <div className="h-56">
          <ResponsiveContainer width="100%" height="100%">
            <PieChart>
              <Pie data={chartData} dataKey="value" nameKey="name" innerRadius={50} outerRadius={80}>
                {chartData.map(function(_,i){ return <Cell key={i} fill={COLORS[i % COLORS.length]} />; })}
              </Pie>
              <Tooltip />
              <Legend />
            </PieChart>
          </ResponsiveContainer>
        </div>
      </div>
    </div>
  );
}

/* =========================
   File: src/components/visuals/PricingTableForm.tsx
   ========================= */
import React, { useState } from "react";
import type { PricingRow } from "../../types/visuals";
import { renderPricingMarkdown } from "../../lib/visualsRender";
import { usePlanStoreV2 } from "../../state/usePlanStore.v2";

export function PricingTableForm({ targetSectionId }:{ targetSectionId: string }){
  const [rows, setRows] = useState<PricingRow[]>([
    { name: "Basic", price: "$19/mo", notes: "For starters" },
    { name: "Pro", price: "$49/mo", notes: "Most popular" },
  ]);
  const update = usePlanStoreV2((s)=> s.updateSection);

  function insert(){
    const md = renderPricingMarkdown(rows);
    const prevAppender = function(p:any){
      var base = (p && (p as any).content) ? (p as any).content + "

" : "";
      return { content: base + md } as any;
    };
    update(targetSectionId, prevAppender as any);
  }

  function change(i:number, k:keyof PricingRow, v:string){ setRows(function(r){ return r.map(function(row,idx){ return idx===i ? { ...row, [k]: v } : row; }); }); }
  function add(){ setRows(function(r){ return r.concat([{ name: "", price: "", notes: "" }]); }); }
  function del(i:number){ setRows(function(r){ return r.filter(function(_,idx){ return idx!==i; }); }); }

  return (
    <div className="grid gap-3">
      <div className="overflow-x-auto">
        <table className="w-full border-collapse border border-border text-sm">
          <thead className="bg-muted">
            <tr>
              <th className="border p-2 text-left">Plan</th>
              <th className="border p-2 text-left">Price</th>
              <th className="border p-2 text-left">Notes</th>
              <th className="border p-2"></th>
            </tr>
          </thead>
          <tbody>
            {rows.map(function(r,i){ return (
              <tr key={i}>
                <td className="border p-2"><input className="w-full outline-none" value={r.name} onChange={function(e){ change(i,'name', e.target.value); }} /></td>
                <td className="border p-2"><input className="w-full outline-none" value={r.price} onChange={function(e){ change(i,'price', e.target.value); }} /></td>
                <td className="border p-2"><input className="w-full outline-none" value={r.notes||""} onChange={function(e){ change(i,'notes', e.target.value); }} /></td>
                <td className="border p-2 text-right"><button className="rounded-md border px-2 py-1" onClick={function(){ del(i); }}>Delete</button></td>
              </tr>
            ); })}
          </tbody>
        </table>
      </div>
      <div className="flex justify-between">
        <button className="rounded-md border px-3 py-1 text-sm" onClick={add}>Add Row</button>
        <button className="rounded-2xl border px-3 py-2" onClick={insert}>Insert into Section</button>
      </div>
    </div>
  );
}

/* =========================
   File: src/components/visuals/CustomChartBuilder.tsx
   ========================= */
import React, { useMemo, useState } from "react";
import { LineChart, Line, BarChart, Bar, PieChart, Pie, Cell, XAxis, YAxis, Tooltip, Legend, ResponsiveContainer } from "recharts";
import Papa from "papaparse";
import * as XLSX from "xlsx";
import { usePlanStoreV2 } from "../../state/usePlanStore.v2";

export function CustomChartBuilder({ targetSectionId }:{ targetSectionId: string }){
  const [title, setTitle] = useState("Custom Chart");
  const [type, setType] = useState("line" as "line"|"bar"|"pie");
  const [data, setData] = useState<any[]>([{ label: "A", value: 10 }, { label: "B", value: 20 }]);
  const [xKey, setXKey] = useState("label");
  const [yKey, setYKey] = useState("value");
  const update = usePlanStoreV2((s)=> s.updateSection);

  const keys = useMemo(function(){ return data.length? Object.keys(data[0]) : ["label","value"]; }, [data]);
  const COLORS = ["#8884d8", "#82ca9d", "#ffc658", "#ff8a65", "#90caf9", "#a5d6a7"];

  function insert(){
    var header = "| " + xKey + " | " + yKey + " |
|---|---:|";
    var body = data.map(function(row){ return "| " + row[xKey] + " | " + row[yKey] + " |"; }).join("
");
    var md = "### " + title + "

**Chart Type:** " + type + "

" + header + "
" + body;
    const prevAppender = function(p:any){
      var base = (p && (p as any).content) ? (p as any).content + "

" : "";
      return { content: base + md } as any;
    };
    update(targetSectionId, prevAppender as any);
  }

  function onCsv(file: File){
    Papa.parse(file, { header: true, skipEmptyLines: true, complete: function(res:any){ if (Array.isArray(res.data) && res.data.length) { setData(res.data); setXKey(Object.keys(res.data[0])[0]); setYKey(Object.keys(res.data[0])[1] || Object.keys(res.data[0])[0]); } } });
  }
  async function onXlsx(file: File){
    const buf = await file.arrayBuffer();
    const wb = XLSX.read(buf, { type: 'array' });
    const ws = wb.Sheets[wb.SheetNames[0]];
    const json: any[] = XLSX.utils.sheet_to_json(ws) as any[];
    if (Array.isArray(json) && json.length){ setData(json as any[]); setXKey(Object.keys(json[0])[0]); setYKey(Object.keys(json[0])[1] || Object.keys(json[0])[0]); }
  }

  return (
    <div className="grid gap-4">
      <div className="grid md:grid-cols-2 gap-3">
        <label className="grid gap-1"><span className="text-sm text-gray-600">Title</span><input className="rounded-xl border p-2" value={title} onChange={function(e){ setTitle(e.target.value); }} /></label>
        <label className="grid gap-1"><span className="text-sm text-gray-600">Chart Type</span>
          <select className="rounded-xl border p-2" value={type} onChange={function(e){ setType(e.target.value as any); }}>
            <option value="line">Line</option>
            <option value="bar">Bar</option>
            <option value="pie">Pie</option>
          </select>
        </label>
        <label className="grid gap-1"><span className="text-sm text-gray-600">X Key</span>
          <select className="rounded-xl border p-2" value={xKey} onChange={function(e){ setXKey(e.target.value); }}>
            {keys.map(function(k){ return <option key={k} value={k}>{k}</option>; })}
          </select>
        </label>
        {type !== 'pie' && (
          <label className="grid gap-1"><span className="text-sm text-gray-600">Y Key</span>
            <select className="rounded-xl border p-2" value={yKey} onChange={function(e){ setYKey(e.target.value); }}>
              {keys.map(function(k){ return <option key={k} value={k}>{k}</option>; })}
            </select>
          </label>
        )}
      </div>

      <div className="grid md:grid-cols-2 gap-4">
        <div className="rounded-2xl border p-3">
          <h5 className="font-medium mb-2">Data Preview</h5>
          <div className="overflow-x-auto">
            <table className="w-full border-collapse border border-border text-sm">
              <thead className="bg-muted">
                <tr>
                  {keys.map(function(k){ return <th key={k} className="border p-2 text-left">{k}</th>; })}
                </tr>
              </thead>
              <tbody>
                {data.slice(0,10).map(function(row,idx){ return (
                  <tr key={idx}>{keys.map(function(k){ return <td key={k} className="border p-2">{String(row[k]||"")}</td>; })}</tr>
                ); })}
              </tbody>
            </table>
          </div>
          <div className="mt-2 flex gap-3 items-center text-sm">
            <label className="cursor-pointer">CSV <input type="file" accept=".csv" className="hidden" onChange={function(e){ if (e.target.files) onCsv(e.target.files[0]); }} /></label>
            <label className="cursor-pointer">Excel <input type="file" accept=".xlsx,.xls" className="hidden" onChange={function(e){ if (e.target.files) onXlsx(e.target.files[0]); }} /></label>
          </div>
        </div>
        <div className="rounded-2xl border p-3">
          <h5 className="font-medium mb-2">Chart Preview</h5>
          <div className="h-56">
            <ResponsiveContainer width="100%" height="100%">
              {type === 'line' && (
                <LineChart data={data}>
                  <XAxis dataKey={xKey} hide/>
                  <YAxis/>
                  <Tooltip/>
                  <Legend/>
                  <Line type="monotone" dataKey={yKey} stroke="#8884d8" dot={false} name={yKey} />
                </LineChart>
              )}
              {type === 'bar' && (
                <BarChart data={data}>
                  <XAxis dataKey={xKey} hide/>
                  <YAxis/>
                  <Tooltip/>
                  <Legend/>
                  <Bar dataKey={yKey} fill="#82ca9d" name={yKey} />
                </BarChart>
              )}
              {type === 'pie' && (
                <PieChart>
                  <Pie data={data} dataKey={yKey} nameKey={xKey} innerRadius={50} outerRadius={80}>
                    {data.map(function(_,i){ return <Cell key={i} fill={COLORS[i % COLORS.length]} />; })}
                  </Pie>
                  <Tooltip/>
                  <Legend/>
                </PieChart>
              )}
            </ResponsiveContainer>
          </div>
        </div>
      </div>

      <div className="flex justify-end">
        <button className="rounded-2xl border px-3 py-2" onClick={insert}>Insert into Section</button>
      </div>
    </div>
  );
}


/* =========================
   EXPORT SNAPSHOTS – Chart → PNG → Insert into Section / Export
   ========================= */

/* =========================
   File: src/lib/snapshot.ts
   ========================= */
import html2canvas from "html2canvas";

export async function captureElementToPng(el: HTMLElement, options?: { scale?: number }): Promise<string> {
  const canvas = await html2canvas(el, {
    backgroundColor: "#ffffff",
    scale: options?.scale ?? 2,
    useCORS: true,
    logging: false,
    windowWidth: el.scrollWidth,
    windowHeight: el.scrollHeight,
  });
  return canvas.toDataURL("image/png"); // data:image/png;base64,...
}

/* =========================
   File: src/components/ChartSnapshotButton.tsx
   ========================= */
import React, { useRef } from "react";
import { captureElementToPng } from "../lib/snapshot";
import { usePlanStoreV2 } from "../state/usePlanStore.v2";

export function ChartSnapshotButton({
  targetSectionId,
  getElement,
  title = "Chart Snapshot",
}: {
  targetSectionId: string;
  getElement: () => HTMLElement | null;
  title?: string;
}){
  const update = usePlanStoreV2((s)=> s.updateSection);
  const busy = useRef(false);

  async function snap(){
    if (busy.current) return; busy.current = true;
    try {
      const el = getElement(); if (!el) return;
      const dataUrl = await captureElementToPng(el, { scale: 2 });
      const md = `### ${title}

![${title}](${dataUrl})`;
      update(targetSectionId, (prev=> ({ content: `${(prev as any)?.content? (prev as any).content + "

" : ""}${md}` })) as any);
    } finally { busy.current = false; }
  }

  return (
    <button className="rounded-md border px-2 py-1 text-sm" onClick={snap}>Insert Snapshot (PNG)</button>
  );
}

/* =========================
   Patch: src/components/visuals/CustomChartBuilder.tsx – add snapshot
   ========================= */
// ...inside component body
// add a ref on chart container and a snapshot button that inserts image markdown

/* replace the Preview card body with a ref'd container and controls */

// (Find the Chart Preview container and wrap with a ref)

/* =========================
   File: src/components/visuals/CustomChartBuilder.tsx (patched excerpt)
   ========================= */
// add imports at top
import React, { useMemo, useRef, useState } from "react";
import { ChartSnapshotButton } from "../ChartSnapshotButton";

// inside component
const chartRef = useRef<HTMLDivElement|null>(null);

// ...existing JSX...
<div className="rounded-2xl border p-3">
  <h5 className="font-medium mb-2">Chart Preview</h5>
  <div className="h-56" ref={chartRef as any}>
    <ResponsiveContainer width="100%" height="100%">
      {type === 'line' && (
        <LineChart data={data}>
          <XAxis dataKey={xKey} hide/>
          <YAxis/>
          <Tooltip/>
          <Legend/>
          <Line type="monotone" dataKey={yKey} stroke="#8884d8" dot={false} name={yKey} />
        </LineChart>
      )}
      {type === 'bar' && (
        <BarChart data={data}>
          <XAxis dataKey={xKey} hide/>
          <YAxis/>
          <Tooltip/>
          <Legend/>
          <Bar dataKey={yKey} fill="#82ca9d" name={yKey} />
        </BarChart>
      )}
      {type === 'pie' && (
        <PieChart>
          <Pie data={data} dataKey={yKey} nameKey={xKey} innerRadius={50} outerRadius={80}>
            {data.map(function(_,i){ return <Cell key={i} fill={COLORS[i % COLORS.length]} />; })}
          </Pie>
          <Tooltip/>
          <Legend/>
        </PieChart>
      )}
    </ResponsiveContainer>
  </div>
  <div className="mt-2 flex justify-end gap-2">
    <button className="rounded-2xl border px-3 py-2" onClick={insert}>Insert as Table</button>
    <ChartSnapshotButton targetSectionId={targetSectionId} getElement={() => chartRef.current} title={title} />
  </div>
</div>

/* =========================
   Patch: src/components/FinanceCharts.tsx – add snapshot controls
   ========================= */
import React, { useRef } from "react";
import { ChartSnapshotButton } from "./ChartSnapshotButton";

export function FinanceCharts(){
  const { outputs, inputs } = useFinanceStoreV2();
  const refRev = useRef<HTMLDivElement|null>(null);
  const refCash = useRef<HTMLDivElement|null>(null);
  const refBe = useRef<HTMLDivElement|null>(null);
  if (!outputs) return null;
  const data = outputs.rows.map(r=> ({ month: r.month, revenue: r.revenue, profit: r.profit, cash: (inputs.startingCash||0) + r.cumulativeProfit }));

  return (
    <div className="grid gap-6">
      <div className="rounded-2xl border p-3">
        <div className="flex items-center justify-between mb-2"><h4 className="font-medium">Revenue vs Profit</h4><ChartSnapshotButton targetSectionId={/* insert into Financial Plan by default */ (document.querySelector('[data-financial-section-id]') as any)?.dataset?.financialSectionId || ''} getElement={()=> refRev.current} title="Revenue vs Profit" /></div>
        <div className="h-56" ref={refRev as any}>
          <ResponsiveContainer width="100%" height="100%">
            <LineChart data={data}>
              <XAxis dataKey="month" hide/>
              <YAxis/>
              <Tooltip formatter={(v:number)=> `${inputs.currency} ${v.toLocaleString()}`}/>
              <Legend />
              <Line type="monotone" dataKey="revenue" stroke="#8884d8" dot={false} name="Revenue" />
              <Line type="monotone" dataKey="profit" stroke="#82ca9d" dot={false} name="Profit" />
            </LineChart>
          </ResponsiveContainer>
        </div>
      </div>

      <div className="rounded-2xl border p-3">
        <div className="flex items-center justify-between mb-2"><h4 className="font-medium">Cash Flow (Ending Cash)</h4><ChartSnapshotButton targetSectionId={(document.querySelector('[data-financial-section-id]') as any)?.dataset?.financialSectionId || ''} getElement={()=> refCash.current} title="Cash Flow (Ending Cash)" /></div>
        <div className="h-56" ref={refCash as any}>
          <ResponsiveContainer width="100%" height="100%">
            <AreaChart data={data}>
              <XAxis dataKey="month" hide/>
              <YAxis/>
              <Tooltip formatter={(v:number)=> `${inputs.currency} ${v.toLocaleString()}`}/>
              <Area type="monotone" dataKey="cash" stroke="#8884d8" fill="#8884d8" fillOpacity={0.2} name="Ending Cash" />
            </AreaChart>
          </ResponsiveContainer>
        </div>
      </div>

      <div className="rounded-2xl border p-3">
        <div className="flex items-center justify-between mb-2"><h4 className="font-medium">Break‑Even (Profit by Month)</h4><ChartSnapshotButton targetSectionId={(document.querySelector('[data-financial-section-id]') as any)?.dataset?.financialSectionId || ''} getElement={()=> refBe.current} title="Break‑Even (Profit by Month)" /></div>
        <div className="h-56" ref={refBe as any}>
          <ResponsiveContainer width="100%" height="100%">
            <BarChart data={outputs.rows}>
              <XAxis dataKey="month" hide/>
              <YAxis/>
              <Tooltip formatter={(v:number)=> `${inputs.currency} ${v.toLocaleString()}`}/>
              <Bar dataKey="profit" name="Profit" fill="#82ca9d" />
            </BarChart>
          </ResponsiveContainer>
        </div>
        {outputs.breakevenMonthIndex!=null && (
          <p className="text-sm mt-2">Breakeven in <strong>{outputs.rows[outputs.breakevenMonthIndex].month}</strong>.</p>
        )}
      </div>
    </div>
  );
}

/* =========================
   Export Note: If your markdown→PDF uses data URLs, these images render automatically.
   If you stream to DOCX, ensure your renderer converts data URLs to embedded media.
   ========================= */


/* =========================
   NEW: Chart snapshots + copy PNG (V2 variants to avoid conflicts)
   ========================= */

/* File: src/lib/snapshot.ts (add copy helper if not present) */
export async function copyDataUrlToClipboard(dataUrl: string) {
  try {
    const res = await fetch(dataUrl);
    const blob = await res.blob();
    // @ts-ignore
    await navigator.clipboard.write([new ClipboardItem({ [blob.type]: blob })]);
    return true;
  } catch (e) {
    console.warn("Clipboard copy failed", e);
    return false;
  }
}

/* =========================
   File: src/components/ChartSnapshotButton.v2.tsx
   ========================= */
import React, { useRef, useState } from "react";
import { captureElementToPng, copyDataUrlToClipboard } from "../lib/snapshot";
import { usePlanStoreV2 } from "../state/usePlanStore.v2";

export function ChartSnapshotButtonV2({
  targetSectionId,
  getElement,
  title = "Chart Snapshot",
}: {
  targetSectionId: string;
  getElement: () => HTMLElement | null;
  title?: string;
}){
  const update = usePlanStoreV2((s)=> s.updateSection);
  const busy = useRef(false);
  const [copied, setCopied] = useState<null|"ok"|"fail">(null);

  async function snapInsert(){
    if (busy.current) return; busy.current = true;
    try {
      const el = getElement(); if (!el) return;
      const dataUrl = await captureElementToPng(el, { scale: 2 });
      const md = `### ${title}

![${title}](${dataUrl})`;
      update(targetSectionId, (prev=> ({ content: `${(prev as any)?.content? (prev as any).content + "

" : ""}${md}` })) as any);
    } finally { busy.current = false; }
  }

  async function snapCopy(){
    const el = getElement(); if (!el) return;
    const dataUrl = await captureElementToPng(el, { scale: 2 });
    const ok = await copyDataUrlToClipboard(dataUrl);
    setCopied(ok ? "ok" : "fail");
    setTimeout(()=> setCopied(null), 2000);
  }

  return (
    <div className="flex items-center gap-2">
      <button className="rounded-md border px-2 py-1 text-sm" onClick={snapInsert}>Insert Snapshot (PNG)</button>
      <button className="rounded-md border px-2 py-1 text-sm" onClick={snapCopy}>{copied==="ok"? "Copied!" : copied==="fail"? "Copy failed" : "Copy PNG"}</button>
    </div>
  );
}

/* =========================
   File: src/components/visuals/MarketSizeForm.v2.tsx
   ========================= */
import React, { useRef, useState } from "react";
import type { MarketSizeData } from "../../types/visuals";
import { renderMarketSizeMarkdown } from "../../lib/visualsRender";
import { usePlanStoreV2 } from "../../state/usePlanStore.v2";
import { PieChart, Pie, Cell, ResponsiveContainer, Tooltip, Legend } from "recharts";
import { ChartSnapshotButtonV2 } from "../ChartSnapshotButton.v2";

export function MarketSizeFormV2({ targetSectionId }:{ targetSectionId: string }){
  const [data, set] = useState<MarketSizeData>({ tam:"", sam:"", som:"", sources:"" });
  const update = usePlanStoreV2((s)=> s.updateSection);
  const chartRef = useRef<HTMLDivElement|null>(null);

  const nums = {
    TAM: Number(String(data.tam).replace(/[^0-9.]/g, "")) || 0,
    SAM: Number(String(data.sam).replace(/[^0-9.]/g, "")) || 0,
    SOM: Number(String(data.som).replace(/[^0-9.]/g, "")) || 0,
  };
  const chartData = [
    { name: "TAM", value: nums.TAM },
    { name: "SAM", value: nums.SAM },
    { name: "SOM", value: nums.SOM },
  ];
  const COLORS = ["#8884d8", "#82ca9d", "#ffc658"];

  function insert(){
    const md = renderMarketSizeMarkdown(data);
    update(targetSectionId, (prev=> ({ content: `${(prev as any)?.content? (prev as any).content + "

" : ""}${md}` })) as any);
  }

  return (
    <div className="grid md:grid-cols-2 gap-4">
      <div className="grid gap-3">
        <label className="grid gap-1"><span className="text-sm text-gray-600">TAM</span><input className="rounded-xl border p-2" placeholder="$ market size" value={data.tam} onChange={(e)=> set({ ...data, tam: e.target.value })} /></label>
        <label className="grid gap-1"><span className="text-sm text-gray-600">SAM</span><input className="rounded-xl border p-2" placeholder="$ market size" value={data.sam} onChange={(e)=> set({ ...data, sam: e.target.value })} /></label>
        <label className="grid gap-1"><span className="text-sm text-gray-600">SOM</span><input className="rounded-xl border p-2" placeholder="$ market size" value={data.som} onChange={(e)=> set({ ...data, som: e.target.value })} /></label>
        <label className="grid gap-1"><span className="text-sm text-gray-600">Sources</span><textarea className="rounded-xl border p-2" rows={2} value={data.sources} onChange={(e)=> set({ ...data, sources: e.target.value })} /></label>
        <div className="flex justify-between items-center">
          <ChartSnapshotButtonV2 targetSectionId={targetSectionId} getElement={()=> chartRef.current} title="TAM / SAM / SOM" />
          <button className="rounded-2xl border px-3 py-2" onClick={insert}>Insert as Text</button>
        </div>
      </div>
      <div className="rounded-2xl border p-3">
        <h5 className="font-medium mb-2">Market Size Visual</h5>
        <div className="h-56" ref={chartRef as any}>
          <ResponsiveContainer width="100%" height="100%">
            <PieChart>
              <Pie data={chartData} dataKey="value" nameKey="name" innerRadius={50} outerRadius={80}>
                {chartData.map(function(_,i){ return <Cell key={i} fill={COLORS[i % COLORS.length]} />; })}
              </Pie>
              <Tooltip />
              <Legend />
            </PieChart>
          </ResponsiveContainer>
        </div>
      </div>
    </div>
  );
}

/* =========================
   File: src/components/visuals/PricingTableForm.v2.tsx
   ========================= */
import React, { useRef, useState } from "react";
import type { PricingRow } from "../../types/visuals";
import { renderPricingMarkdown } from "../../lib/visualsRender";
import { usePlanStoreV2 } from "../../state/usePlanStore.v2";
import { ChartSnapshotButtonV2 } from "../ChartSnapshotButton.v2";

export function PricingTableFormV2({ targetSectionId }:{ targetSectionId: string }){
  const [rows, setRows] = useState<PricingRow[]>([
    { name: "Basic", price: "$19/mo", notes: "For starters" },
    { name: "Pro", price: "$49/mo", notes: "Most popular" },
  ]);
  const update = usePlanStoreV2((s)=> s.updateSection);
  const tableRef = useRef<HTMLDivElement|null>(null);

  function insert(){
    const md = renderPricingMarkdown(rows);
    update(targetSectionId, (prev=> ({ content: `${(prev as any)?.content? (prev as any).content + "

" : ""}${md}` })) as any);
  }

  function change(i:number, k:keyof PricingRow, v:string){ setRows(r=> r.map((row,idx)=> idx===i ? { ...row, [k]: v } : row)); }
  function add(){ setRows(r=> [...r, { name: "", price: "", notes: "" }]); }
  function del(i:number){ setRows(r=> r.filter((_,idx)=> idx!==i)); }

  return (
    <div className="grid gap-3">
      <div className="overflow-x-auto" ref={tableRef as any}>
        <table className="w-full border-collapse border border-border text-sm">
          <thead className="bg-muted">
            <tr>
              <th className="border p-2 text-left">Plan</th>
              <th className="border p-2 text-left">Price</th>
              <th className="border p-2 text-left">Notes</th>
              <th className="border p-2"></th>
            </tr>
          </thead>
          <tbody>
            {rows.map((r,i)=> (
              <tr key={i}>
                <td className="border p-2"><input className="w-full outline-none" value={r.name} onChange={(e)=> change(i,'name', e.target.value)} /></td>
                <td className="border p-2"><input className="w-full outline-none" value={r.price} onChange={(e)=> change(i,'price', e.target.value)} /></td>
                <td className="border p-2"><input className="w-full outline-none" value={r.notes||""} onChange={(e)=> change(i,'notes', e.target.value)} /></td>
                <td className="border p-2 text-right"><button className="rounded-md border px-2 py-1" onClick={()=> del(i)}>Delete</button></td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
      <div className="flex justify-between items-center">
        <button className="rounded-md border px-3 py-1 text-sm" onClick={add}>Add Row</button>
        <div className="flex gap-2">
          <ChartSnapshotButtonV2 targetSectionId={targetSectionId} getElement={()=> tableRef.current} title="Pricing Table" />
          <button className="rounded-2xl border px-3 py-2" onClick={insert}>Insert as Table</button>
        </div>
      </div>
    </div>
  );
}

/* =========================
   STEP 5 – Cover Designs & Theme Packs (scaffold)
   ========================= */

/* File: src/types/theme.ts */
export type CoverStyle = "Gradient" | "Minimal" | "Photo" | "Pattern";
export interface CoverTheme {
  id: string; name: string; style: CoverStyle;
  primary: string; secondary?: string; accent?: string;
  fontHead?: string; fontBody?: string; pattern?: "dots"|"grid"|"waves"|"none";
}

/* File: src/lib/coverThemes.ts */
import type { CoverTheme } from "../types/theme";
export const COVER_THEMES: CoverTheme[] = [
  { id: "aurora", name: "Aurora", style: "Gradient", primary: "#6D28D9", secondary: "#06B6D4", accent: "#F59E0B", fontHead: "Inter, ui-sans-serif", fontBody: "Inter, ui-sans-serif", pattern: "waves" },
  { id: "noir", name: "Noir", style: "Minimal", primary: "#111827", secondary: "#374151", accent: "#10B981", fontHead: "Manrope, ui-sans-serif", fontBody: "Manrope, ui-sans-serif", pattern: "none" },
  { id: "sunset", name: "Sunset", style: "Gradient", primary: "#EF4444", secondary: "#F59E0B", accent: "#FDE68A", fontHead: "Poppins, ui-sans-serif", fontBody: "Poppins, ui-sans-serif", pattern: "dots" },
  { id: "paper", name: "Paper", style: "Pattern", primary: "#2563EB", secondary: "#93C5FD", accent: "#F8FAFC", fontHead: "Source Sans Pro, ui-sans-serif", fontBody: "Source Sans Pro, ui-sans-serif", pattern: "grid" },
];

/* File: src/components/CoverDesigner.tsx */
import React, { useRef, useState } from "react";
import { COVER_THEMES } from "../lib/coverThemes";
import type { CoverTheme } from "../types/theme";
import { captureElementToPng } from "../lib/snapshot";

export function CoverDesigner({ onSaveImage }:{ onSaveImage?: (dataUrl:string)=> void }){
  const [theme, setTheme] = useState<CoverTheme>(COVER_THEMES[0]);
  const [title, setTitle] = useState("Business Plan");
  const [subtitle, setSubtitle] = useState("for Your Company");
  const [logoUrl, setLogoUrl] = useState<string>("");
  const ref = useRef<HTMLDivElement|null>(null);

  async function exportPng(){
    if (!ref.current) return;
    const png = await captureElementToPng(ref.current, { scale: 2 });
    onSaveImage?.(png);
  }

  return (
    <div className="grid gap-4">
      <div className="grid md:grid-cols-2 gap-3">
        <label className="grid gap-1"><span className="text-sm text-gray-600">Theme</span>
          <select className="rounded-xl border p-2" value={theme.id} onChange={(e)=> setTheme(COVER_THEMES.find(t=> t.id===e.target.value) || COVER_THEMES[0])}>
            {COVER_THEMES.map(t=> <option key={t.id} value={t.id}>{t.name}</option>)}
          </select>
        </label>
        <label className="grid gap-1"><span className="text-sm text-gray-600">Title</span><input className="rounded-xl border p-2" value={title} onChange={(e)=> setTitle(e.target.value)} /></label>
        <label className="grid gap-1"><span className="text-sm text-gray-600">Subtitle</span><input className="rounded-xl border p-2" value={subtitle} onChange={(e)=> setSubtitle(e.target.value)} /></label>
        <label className="grid gap-1"><span className="text-sm text-gray-600">Logo URL (optional)</span><input className="rounded-xl border p-2" value={logoUrl} onChange={(e)=> setLogoUrl(e.target.value)} placeholder="https://..." /></label>
      </div>

      <div className="rounded-2xl border p-3">
        <h5 className="font-medium mb-2">Cover Preview</h5>
        <div ref={ref as any} className="relative w-full" style={{ aspectRatio: "1.4142 / 1" }}>
          <CoverCanvas theme={theme} title={title} subtitle={subtitle} logoUrl={logoUrl} />
        </div>
        <div className="mt-2 flex justify-end">
          <button className="rounded-2xl border px-3 py-2" onClick={exportPng}>Export Cover PNG</button>
        </div>
      </div>
    </div>
  );
}

function CoverCanvas({ theme, title, subtitle, logoUrl }:{ theme: CoverTheme; title: string; subtitle: string; logoUrl?: string }){
  const grad = theme.style === "Gradient" ? { backgroundImage: `linear-gradient(135deg, ${theme.primary}, ${theme.secondary})` } : {} as any;
  const bg = theme.style === "Minimal" ? { backgroundColor: theme.primary } : {} as any;
  const patternStyle: any = theme.pattern === "dots" ? { backgroundImage: "radial-gradient(currentColor 1px, transparent 1px)", backgroundSize: "16px 16px", color: "rgba(255,255,255,0.15)" } : theme.pattern === "grid" ? { backgroundImage: "linear-gradient(rgba(255,255,255,0.12) 1px, transparent 1px), linear-gradient(90deg, rgba(255,255,255,0.12) 1px, transparent 1px)", backgroundSize: "20px 20px" } : {};
  return (
    <div className="w-full h-full rounded-xl overflow-hidden text-white p-10 flex flex-col justify-between" style={{ ...grad, ...bg }}>
      <div className="absolute inset-0 pointer-events-none" style={patternStyle}></div>
      <div className="flex items-center justify-between relative">
        <div className="text-sm opacity-80" style={{ fontFamily: theme.fontBody }}>IBrandBiz</div>
        {logoUrl && <img src={logoUrl} alt="logo" className="h-10 w-auto object-contain" />}
      </div>
      <div className="relative" style={{ fontFamily: theme.fontHead }}>
        <h1 className="text-4xl md:text-6xl font-bold leading-tight">{title}</h1>
        <p className="text-lg md:text-2xl mt-3 opacity-90">{subtitle}</p>
      </div>
      <div className="flex items-center justify-between relative" style={{ fontFamily: theme.fontBody }}>
        <div className="text-sm opacity-80">{new Date().toLocaleDateString()}</div>
        <div className="text-sm opacity-80">www.ibrandbiz.com</div>
      </div>
    </div>
  );
}

/* File: src/pages/Builder.v5.tsx (include Cover Designer) */
import React from "react";
import { OutlineEditorV2 } from "../components/OutlineEditor.v2";
import { usePlanStoreV2 } from "../state/usePlanStore.v2";
import { SectionEditor } from "../components/SectionEditor";
import { FinancialsPanel } from "../components/FinancialsPanel";
import { TemplatesPanel } from "../components/TemplatesPanel";
import { CoverDesigner } from "../components/CoverDesigner";

export default function BuilderV5(){
  const sections = usePlanStoreV2((s)=> s.sections);
  const first = sections[0];
  return (
    <div className="grid 2xl:grid-cols-[1.1fr_0.9fr] gap-8 p-6">
      <div className="space-y-6">
        <h2 className="text-xl font-semibold">Outline</h2>
        <OutlineEditorV2 />
        {first && <div className="mt-6"><SectionEditor sectionId={first.id} /></div>}
      </div>
      <div className="space-y-8">
        <div>
          <h2 className="text-xl font-semibold mb-2">Cover & Theme</h2>
          <CoverDesigner />
        </div>
        <div>
          <h2 className="text-xl font-semibold mb-2">Visual Templates</h2>
          <TemplatesPanel />
        </div>
        <div>
          <h2 className="text-xl font-semibold mb-2">Financial Tools</h2>
          <FinancialsPanel />
        </div>
      </div>
    </div>
  );
}

/* NOTE: To use snapshot-capable forms, swap imports in TemplatesPanel to MarketSizeFormV2 and PricingTableFormV2 if desired. */

/* =========================
   BRAND KIT – types, store, hooks, integration
   ========================= */

/* =========================
   File: src/types/brand.ts
   ========================= */
export interface BrandKit {
  name?: string;
  primary: string;   // hex
  secondary?: string; // hex
  accent?: string;   // hex
  logoUrl?: string;
  fontHead?: string; // CSS font stack
  fontBody?: string; // CSS font stack
}

/* =========================
   File: src/state/useBrandKit.ts
   ========================= */
import { create } from "zustand";
import type { BrandKit } from "../types/brand";

const DEFAULT_BRAND: BrandKit = {
  name: "IBrandBiz",
  primary: "#6D28D9",
  secondary: "#06B6D4",
  accent: "#F59E0B",
  logoUrl: "",
  fontHead: "Inter, ui-sans-serif",
  fontBody: "Inter, ui-sans-serif",
};

export interface BrandKitStore {
  brand: BrandKit;
  setBrand: (bk: Partial<BrandKit>) => void;
  resetBrand: () => void;
}

export const useBrandKit = create<BrandKitStore>((set) => ({
  brand: DEFAULT_BRAND,
  setBrand: (bk) => set((s) => ({ brand: { ...s.brand, ...bk } })),
  resetBrand: () => set({ brand: DEFAULT_BRAND }),
}));

/* =========================
   File: src/lib/brand.ts
   ========================= */
import type { BrandKit } from "../types/brand";

export function getBrandPalette(brand: BrandKit){
  // Return a safe palette for charts (fallbacks included)
  const p = brand.primary || "#6D28D9";
  const s = brand.secondary || "#82ca9d";
  const a = brand.accent || "#ffc658";
  const alt1 = shadeHex(p, -12);
  const alt2 = shadeHex(s, -12);
  const alt3 = shadeHex(a, -12);
  return [p, s, a, alt1, alt2, alt3];
}

export function shadeHex(hex: string, percent: number){
  try{
    const h = hex.replace("#", "");
    const num = parseInt(h, 16);
    const r = Math.min(255, Math.max(0, (num >> 16) + Math.round(2.55*percent)));
    const g = Math.min(255, Math.max(0, ((num >> 8) & 0x00FF) + Math.round(2.55*percent)));
    const b = Math.min(255, Math.max(0, (num & 0x0000FF) + Math.round(2.55*percent)));
    return `#${(1<<24 | (r<<16) | (g<<8) | b).toString(16).slice(1)}`;
  } catch { return hex; }
}

/* =========================
   File: src/components/BrandKitPanel.tsx
   ========================= */
import React from "react";
import { useBrandKit } from "../state/useBrandKit";

export function BrandKitPanel(){
  const { brand, setBrand, resetBrand } = useBrandKit();
  const bind = (k: keyof typeof brand) => ({ value: (brand as any)[k] || "", onChange: (e:any)=> setBrand({ [k]: e.target.value } as any) });
  return (
    <div className="grid gap-3">
      <div className="grid md:grid-cols-2 gap-3">
        <label className="grid gap-1"><span className="text-sm text-gray-600">Brand Name</span><input className="rounded-xl border p-2" {...bind("name")} placeholder="Your brand" /></label>
        <label className="grid gap-1"><span className="text-sm text-gray-600">Logo URL</span><input className="rounded-xl border p-2" {...bind("logoUrl")} placeholder="https://..." /></label>
        <label className="grid gap-1"><span className="text-sm text-gray-600">Primary Color</span><input type="color" className="rounded-xl border p-2 h-10" {...bind("primary")} /></label>
        <label className="grid gap-1"><span className="text-sm text-gray-600">Secondary Color</span><input type="color" className="rounded-xl border p-2 h-10" {...bind("secondary")} /></label>
        <label className="grid gap-1"><span className="text-sm text-gray-600">Accent Color</span><input type="color" className="rounded-xl border p-2 h-10" {...bind("accent")} /></label>
        <label className="grid gap-1"><span className="text-sm text-gray-600">Heading Font</span><input className="rounded-xl border p-2" {...bind("fontHead")} placeholder="Inter, ui-sans-serif" /></label>
        <label className="grid gap-1"><span className="text-sm text-gray-600">Body Font</span><input className="rounded-xl border p-2" {...bind("fontBody")} placeholder="Inter, ui-sans-serif" /></label>
      </div>
      <div className="flex justify-between">
        <button className="rounded-md border px-3 py-2 text-sm" onClick={resetBrand}>Reset</button>
        <span className="text-xs text-muted-foreground">Applied automatically to covers & charts</span>
      </div>
    </div>
  );
}

/* =========================
   Patch: src/components/CoverDesigner.tsx – auto-apply brand kit
   ========================= */
import { useBrandKit } from "../state/useBrandKit";

// inside CoverDesigner component
const brand = useBrandKit((s)=> s.brand);
// hydrate theme defaults from brand if present
React.useEffect(()=>{
  setTheme((t)=> ({
    ...t,
    primary: brand.primary || t.primary,
    secondary: brand.secondary || t.secondary,
    accent: brand.accent || t.accent,
    fontHead: brand.fontHead || t.fontHead,
    fontBody: brand.fontBody || t.fontBody,
  }));
  if (!logoUrl && brand.logoUrl) setLogoUrl(brand.logoUrl);
  if (title === "Business Plan" && brand.name) setTitle(`${brand.name} – Business Plan`);
}, [brand]);

/* =========================
   Patch: src/components/visuals/CustomChartBuilder.tsx – use brand colors
   ========================= */
import { useBrandKit } from "../../state/useBrandKit";
import { getBrandPalette } from "../../lib/brand";

// inside component
const brand = useBrandKit((s)=> s.brand);
const palette = getBrandPalette(brand);

// replace hard-coded colors with palette
// Line stroke -> palette[0], Bar fill -> palette[1], Pie cells cycle palette
// Example (within JSX):
// <Line ... stroke={palette[0]} />
// <Bar ... fill={palette[1]} />
// {data.map((_,i)=> <Cell key={i} fill={palette[i % palette.length]} />)}

/* =========================
   Patch: src/components/FinanceCharts.tsx – use brand colors
   ========================= */
import { useBrandKit } from "../state/useBrandKit";
import { getBrandPalette } from "../lib/brand";

// inside component
const brand = useBrandKit((s)=> s.brand);
const palette = getBrandPalette(brand);
// replace chart colors
// revenue: palette[0], profit: palette[1], cash area: palette[0] with 0.2 fill

/* =========================
   File: src/pages/Builder.v6.tsx – add Brand Kit panel
   ========================= */
import React from "react";
import { OutlineEditorV2 } from "../components/OutlineEditor.v2";
import { usePlanStoreV2 } from "../state/usePlanStore.v2";
import { SectionEditor } from "../components/SectionEditor";
import { FinancialsPanel } from "../components/FinancialsPanel";
import { TemplatesPanel } from "../components/TemplatesPanel";
import { CoverDesigner } from "../components/CoverDesigner";
import { BrandKitPanel } from "../components/BrandKitPanel";

export default function BuilderV6(){
  const sections = usePlanStoreV2((s)=> s.sections);
  const first = sections[0];
  return (
    <div className="grid 2xl:grid-cols-[1.1fr_0.9fr] gap-8 p-6">
      <div className="space-y-6">
        <h2 className="text-xl font-semibold">Outline</h2>
        <OutlineEditorV2 />
        {first && <div className="mt-6"><SectionEditor sectionId={first.id} /></div>}
      </div>
      <div className="space-y-8">
        <div>
          <h2 className="text-xl font-semibold mb-2">Brand Kit</h2>
          <BrandKitPanel />
        </div>
        <div>
          <h2 className="text-xl font-semibold mb-2">Cover & Theme</h2>
          <CoverDesigner />
        </div>
        <div>
          <h2 className="text-xl font-semibold mb-2">Visual Templates</h2>
          <TemplatesPanel />
        </div>
        <div>
          <h2 className="text-xl font-semibold mb-2">Financial Tools</h2>
          <FinancialsPanel />
        </div>
      </div>
    </div>
  );
}

/* =========================
   Notes
   - BrandKitPanel lets users set colors, fonts, logo once.
   - CoverDesigner and all charts consume brand kit automatically.
   - getBrandPalette() guarantees pleasant chart hues and fallback colors.
   ========================= */

